#include "language.h"
#ifndef LATTICE
#define DICE
#endif

#define INCLUDE3_0

#include "exec/types.h"
#include "exec/memory.h"
#include "graphics/gfx.h"
#include "graphics/gfxbase.h"
#include "graphics/videocontrol.h"
#include "graphics/gfxmacros.h"
#include "intuition/screens.h"
#include "libraries/dos.h"
#include "stddef.h"

#ifdef LATTICE
#include "proto/exec.h"
#include "proto/graphics.h"
#include "proto/intuition.h"
#include "proto/dos.h"
#endif

#ifdef DICE
#include "clib/exec_protos.h"
#include "clib/graphics_protos.h"
#include "clib/intuition_protos.h"
#include "clib/dos_protos.h"
#endif

#include "qiff.h"

#define EHB EXTRA_HALFBRITE

#ifndef max
#define max(a,b) ((a)>(b)?(a):(b))
#define min(a,b) ((a)<=(b)?(a):(b))
#endif

#ifdef LATTICE
void __asm USE_FRAME(register __a0 struct BitMap *B,
                     register __a2 struct Frame *F,
                     register __d6 long TrueBytesPerRow);

void __asm FILL_BITMAP(register __a0 unsigned char *nextbyte,
                        register __a1 unsigned char **planes,
                        register __d0 unsigned long depth,
                        register __d1 unsigned long lineBytes,
                        register __d2 unsigned long bytesPerRow,
                        register __d3 unsigned long rowOffset,
                        register __d4 unsigned long lines);

void __asm UNPACK_BITMAP(register __a0 unsigned char *nextbyte,
                           register __a1 unsigned char **planes,
                           register __a2 unsigned char *NULLBuf,
                           register __d0 unsigned long depth,
                           register __d1 unsigned long lineBytes,
                           register __d2 unsigned long bytesPerRow,
                           register __d3 unsigned long rowOffset,
                           register __d4 unsigned long lines);
#endif

#ifdef DICE
/* B -> a0, F -> a1, TrueBytesPerRow -> d0 */
__regargs void USE_FRAME_DCC(struct BitMap *B,\
                             struct Frame *F,\
                             long TrueBytesPerRow);
#define USE_FRAME USE_FRAME_DCC

/* nextbyte -> a0, planes -> a1, depth -> d0, lineBytes -> d1 */
/* bytesPerRow -> d2, rowOffset -> d3, lines -> d4 */
void FILL_BITMAP(unsigned char *nextbyte,
                  unsigned char **planes,
                  unsigned long depth,
                  unsigned long lineBytes,
                  unsigned long bytesPerRow,
                  unsigned long rowOffset,
                  unsigned long lines);

#define FILL_BITMAP FILL_BITMAP_DCC

/* nextbyte -> a0, planes -> a1, NULLBuf -> a2, depth -> d0, lineBytes -> d1 */
/* bytesPerRow -> d2, rowOffset -> d3, lines -> d4 */
void UNPACK_BITMAP(unsigned char *nextbyte,
                  unsigned char **planes,
                  unsigned char *NULLBuf,
                  unsigned long depth,
                  unsigned long lineBytes,
                  unsigned long bytesPerRow,
                  unsigned long rowOffset,
                  unsigned long lines);

#define UNPACK_BITMAP UNPACK_BITMAP_DCC

#endif

#ifdef INCLUDE3_0
#define TransparencyBits LowColorBits
#else
#define BMB_CLEAR 0
#define BMB_DISPLAYABLE 1
#define BMB_INTERLEAVED 2
#define BMB_STANDARD 3
#define BMB_MINPLANES 4

#define BMF_CLEAR (1l<<BMB_CLEAR)
#define BMF_DISPLAYABLE (1l<<BMB_DISPLAYABLE)
#define BMF_INTERLEAVED (1l<<BMB_INTERLEAVED)
#define BMF_STANDARD (1l<<BMB_STANDARD)
#define BMF_MINPLANES (1l<<BMB_MINPLANES)
#endif

extern struct GfxBase *GfxBase;
extern struct IntuitionBase *IntuitionBase;

extern ULONG SystemVer;

extern BOOL TestMessage(struct IntuiMessage *IM);

struct IntuiMessage IMCopy;

/* ---------------------------------------------------------------- */

/*
#include "hardware/custom.h"
#include "hardware/dmabits.h"

/ *
This is clear system way, but it needs LargeData Model compile
extern struct Custom custom;
* /

#define custom (*((struct Custom *)0xdff000))
*/

/* ----------------------- IFF procedures ------------------------- */

#define IFF_NoError              0
#define IFF_End                  1
#define IFF_NoMemory             2
#define IFF_NoFile               3
#define IFF_NoILBM_ANIM          4
#define IFF_FileCorrupt          5
#define IFF_NoCAMG               6
#define IFF_NoCMAP               7
#define IFF_NoBMHD               8
#define IFF_NoBODY               9
#define IFF_NoAMIGADepth         10
#define IFF_BadCompression       11
#define IFF_DoubleCAMG           12
#define IFF_DoubleCMAP           13
#define IFF_DoubleBMHD           14
#define IFF_DoubleBODY           15
#define IFF_NoANHD               16
#define IFF_NoDLTA               17
#define IFF_DoubleANHD           18
#define IFF_DoubleDLTA           19
#define IFF_DoubleDPAN           20
#define IFF_BadDPAN              21
#define IFF_BadDLTA              22
#define IFF_BadBMHD              23
#define IFF_BadCAMG              24
#define IFF_BadBODY              25
#define IFF_InterleaveChanged    26
#define IFF_BadInterleave        27
#define IFF_NotEnoughFrames      28
#define IFF_TooMuchFrames        29
#define IFF_BadAnimCompression   30
#define IFF_BadAnimModes         31

#define IFF_UnknownScreenMode    32
#define IFF_BadDims              33
#define IFF_NoScreen             34

char *IFFMsg[]=
{
#ifdef GERMAN
   "IFF ILBM korekt gelesen",
#else
   "IFF ILBM readed correctly",
#endif
#ifdef GERMAN
   "IFF ILBM gelesen zum Ende der Datei",
#else
   "IFF ILBM readed to the end of file",
#endif
#ifdef GERMAN
   "nicht genug Speicher",
#else
   "no memory",
#endif
#ifdef GERMAN
   "keine solche Datei",
#else
   "no such a file",
#endif
#ifdef GERMAN
   "keine IFF ILBM oder IFF ANIM Datei",
#else
   "not an IFF ILBM nor IFF ANIM file",
#endif
#ifdef GERMAN
   "Datei hat felerhafte Struktur",
#else
   "file has corrupt structure",
#endif
#ifdef GERMAN
   "kein CAMG Chunk",
#else
   "missing CAMG chunk",
#endif
#ifdef GERMAN
   "kein CMAP Chunk",
#else
   "missing CMAP chunk",
#endif
#ifdef GERMAN
   "kein BMHD Chunk",
#else
   "missing BMHD chunk",
#endif
#ifdef GERMAN
   "kein BODY Chunk",
#else
   "missing BODY chunk",
#endif
#ifdef GERMAN
   "kein AMIGA ILBM (ANIM) - zu viele Bitebenen",
#else
   "not an AMIGA ILBM nor AMIGA ANIM - depth exceeded",
#endif
#ifdef GERMAN
   "nicht bekannte BODY Kompresion",
#else
   "unknown BODY compression used",
#endif
#ifdef GERMAN
   "zweifaches CAMG Chunk",
#else
   "double CAMG chunk",
#endif
#ifdef GERMAN
   "zweifaches CMAP Chunk",
#else
   "double CMAP chunk",
#endif
#ifdef GERMAN
   "zweifaches BMHD Chunk",
#else
   "double BMHD chunk",
#endif
#ifdef GERMAN
   "zweifaches BODY Chunk",
#else
   "double BODY chunk",
#endif
#ifdef GERMAN
   "kein ANHD Chunk",
#else
   "missing ANHD chunk",
#endif
#ifdef GERMAN
   "kein DLTA Chunk",
#else
   "missing DLTA chunk",
#endif
#ifdef GERMAN
   "zweifaches ANHD Chunk",
#else
   "double ANHD chunk",
#endif
#ifdef GERMAN
   "zweifaches DLTA Chunk",
#else
   "double DLTA chunk",
#endif
#ifdef GERMAN
   "zweifaches DPAN Chunk",
#else
   "double DPAN chunk",
#endif
#ifdef GERMAN
   "DPAN in animation Frames",
#else
   "DPAN in animation frames",
#endif
#ifdef GERMAN
   "DLTA in erste Frame",
#else
   "DLTA in first frame",
#endif
#ifdef GERMAN
   "BMHD in animation Frames",
#else
   "BMHD in animation frames",
#endif
#ifdef GERMAN
   "CAMG in animation Frames",
#else
   "CAMG in animation frames",
#endif
#ifdef GERMAN
   "BODY in animation Frames",
#else
   "BODY in animation frames",
#endif
#ifdef GERMAN
   "animation Interleave gendert",
#else
   "animation interleave changed",
#endif
#ifdef GERMAN
   "ungltiges animation Interleave",
#else
   "bad animation interleave",
#endif
#ifdef GERMAN
   "nicht genug Frames",
#else
   "not enough frames",
#endif
#ifdef GERMAN
   "zu viele Frames",
#else
   "too much frames",
#endif
#ifdef GERMAN
   "nicht unterstztes anim Kompresion",
#else
   "unsupported anim compression used",
#endif
#ifdef GERMAN
   "nicht unterstztes anim Mode",
#else
   "unsupported anim modes used",
#endif
#ifdef GERMAN
   "nicht bekanntes bildschirm Mode",
#else
   "Unknown screen mode",
#endif
#ifdef GERMAN
   "ungltige Bilddimensionen",
#else
   "Bad picture dimensions",
#endif
#ifdef GERMAN
   "kann Bildschirm nicht ffnen"
#else
   "Can't open screen"
#endif
};

#define idFORM 'FORM'
#define idILBM 'ILBM'
#define idANIM 'ANIM'
#define idBMHD 'BMHD'
#define idDPAN 'DPAN'
#define idCMAP 'CMAP'
#define idCAMG 'CAMG'
#define idBODY 'BODY'
#define idANHD 'ANHD'
#define idDLTA 'DLTA'

struct ColorRegister
{
   unsigned char red,green,blue;
};

struct IFFHeader
{
   long group;
   long size;
   long type;
};

struct ChunkHeader
{
   long ID;
   long size;
};

struct CAMG
{
   unsigned long ViewModes;
};

struct CMAP
{
   struct ColorRegister cregs;
};

struct BMHD
{
   unsigned short w,h;
   short x,y;
   unsigned char nPlanes;
   unsigned char masking;
   unsigned char compression;
   unsigned char pad1;
   unsigned short transparentColor;
   unsigned char xAspect,yAspect;
   short pageWidth,pageHeight;
};

#define mskNone                  0
#define mskHasMask               1
#define mskHasTransparentColor   2
#define mskLasso                 3

struct BODY
{
   unsigned char bytes;
};

struct DPAN
{
   UWORD version;
   UWORD nframes;
   ULONG flags;
};

struct ANHD
{
   UBYTE operation;
   UBYTE mask;
   UWORD w,h;
   WORD x,y;
   ULONG abstime;
   ULONG reltime;
   UBYTE interleave;
   UBYTE pad0;
   ULONG bits;
   UBYTE pad[16];
};

struct DLTA
{
   UBYTE *deltas[16];
   UBYTE _deltas;
};

struct MyDLTA
{
   ULONG *deltas[16];
   UBYTE _deltas;
};

struct Frame
{
   struct Frame *NextFrame;
   ULONG reltime;
   ULONG bits;
   union Data
   {
      struct BODY *BODY;
      struct DLTA *DLTA;
      struct MyDLTA *MyDLTA;
   } Data;
   ULONG DataSize;
   struct ColorMap *ColorMap;
};

struct Animation
{
   struct Screen *Screen;
   struct Window *Window;
   APTR ScreenColorMapData;
   APTR ScreenColorMapLowData;
   UWORD ScreenColorCount;
   BOOL CM39;

   struct BitMap BitMap,*rBitMap;
   unsigned short width,height;
   unsigned short bm_width,bm_height;
   unsigned char depth,masking;
   unsigned long ViewModes;
   short pWidth,pHeight;
   unsigned char compression,pad1;
   struct Frame InitialFrame;

   ULONG flags;

   unsigned short frames;
   struct Frame *FirstFrame;
   UBYTE interleave,pad2;
   struct BitMap AnimMap,*rAnimMap;

   short OldDxOffset,OldDyOffset;
   short OldSLeft,OldSTop;
};

#define PIF_ANIM     1
#define PIF_LOOP     2
#define PIF_CAMG     4

/*-----------------------------------------------------------*/

ULONG IFFErr=IFF_NoError;

#ifdef LATTICE
USHORT __chip NoPointer[]= { 0,0,0,0 };

#define GetNoPointer()
#define UnGetNoPointer()
#else
USHORT *NoPointer=NULL;

void GetNoPointer(void)
{
   NoPointer=(SHORT *)AllocMem(4,MEMF_CHIP | MEMF_CLEAR);
}

void UnGetNoPointer(void)
{
   if (NoPointer) FreeMem(NoPointer,4);
}
#endif

/*-----------------------------------------------------------*/

/*
void printfID(long ID)
{
   char IDstr[6];

   IDstr[4]='\0';

   *((long *)IDstr)=ID;
   printf("%s",IDstr);
}

void printfChunkHeader(struct ChunkHeader *CH)
{
   printfID(CH->ID); printf(" %8ld",CH->size);
}

void printfFORM(struct IFFHeader *IFFH)
{
   printfChunkHeader((struct ChunkHeader *)IFFH);
   printf(" ");
   printfID(IFFH->type);
}

void printfIFFLevel(ULONG l)
{
   while(l--) printf(".");
}
*/

void FillBitMap(struct BitMap *B,USHORT W,USHORT H,unsigned char D,
               BOOL hasMask,char *mask,unsigned char C,struct BODY *BODY)
{
   register unsigned short i,w,h;
   register long BytesPerRow,RowOffset,LineBytes;
   unsigned char *nextbyte,depth,*NULLBuf;
   unsigned char **Planes;

   nextbyte=&(BODY->bytes);
   BytesPerRow=B->BytesPerRow;
   LineBytes=((W+15)>>4)<<1;
   depth=D; w=W; h=H;

   if (hasMask) depth++;

   if (!(NULLBuf=AllocMem(LineBytes+depth<<2,MEMF_PUBLIC | MEMF_CLEAR)))
   { IFFErr=IFF_NoMemory; return; }

   Planes=(unsigned char **)(NULLBuf+LineBytes);

   for (i=0;i<D;i++) Planes[i]=B->Planes[i];
   if (hasMask) Planes[D]=mask;

   RowOffset=0;
   switch(C)
   {
      case 0:
         FILL_BITMAP(nextbyte,Planes,depth,LineBytes,BytesPerRow,0,h);
         break;
      case 1:
         UNPACK_BITMAP(nextbyte,Planes,NULLBuf,depth,LineBytes,BytesPerRow,0,h);
         break;
      default: IFFErr=IFF_BadCompression; break;
   }

   FreeMem(NULLBuf,LineBytes+depth<<2);

   return;
}

void DoBMHD(struct Animation *Anim,struct Frame *Frame,struct BMHD *BMHD,
            struct ChunkHeader *CH)
{
   Anim->width=BMHD->w;
   Anim->height=BMHD->h;
   Anim->bm_width=BMHD->w;
   Anim->bm_height=BMHD->h;
   Anim->depth=BMHD->nPlanes;
   Anim->pWidth=BMHD->pageWidth;
   Anim->pHeight=BMHD->pageHeight;
   Anim->compression=BMHD->compression;
   Anim->masking=BMHD->masking;

   if (Anim->depth>8) IFFErr=IFF_NoAMIGADepth;
}

void DoCAMG(struct Animation *Anim,struct Frame *Frame,struct CAMG *CAMG,
            struct ChunkHeader *CH)
{
   Anim->ViewModes=CAMG->ViewModes;
}

void FillColorMap4(struct ColorMap *CM,struct ColorRegister *cr,ULONG size)
{
   register SHORT i;
   register UBYTE r,g,b,*c;

   c=(UBYTE *)cr;
   for (i=0; i<size; i++, cr++)
   {
      r=*c++; g=*c++; b=*c++;
      r=(r & 0xf0)>>4; g=(g & 0xf0)>>4; b=(b & 0xf0)>>4;
      SetRGB4CM(CM,i,r,g,b);
   }
}

void FillColorMap32(struct ColorMap *CM,struct ColorRegister *cr,ULONG size)
{
   register ULONG i,r,g,b;
   register UBYTE *c;

   c=(UBYTE *)cr;
   for (i=0; i<size; i++, cr++)
   {
      r=*c++; g=*c++; b=*c++;
      r=(r<<24)+(r<<16)+(r<<8)+r;
      g=(g<<24)+(g<<16)+(g<<8)+g;
      b=(b<<24)+(b<<16)+(b<<8)+b;
      SetRGB32CM(CM,i,r,g,b);
   }
}

void DoCMAP(struct Animation *Anim,struct Frame *Frame,struct CMAP *CMAP,
            struct ChunkHeader *CH)
{
   ULONG size;

   size=CH->size/3;

   if (!size) return;

   if (!(Frame->ColorMap=GetColorMap(max(size,20))))
   { IFFErr=IFF_NoMemory; return; }

   if ((SystemVer>=39) && (Anim->CM39))
      FillColorMap32(Frame->ColorMap,&(CMAP->cregs),size);
   else
      FillColorMap4(Frame->ColorMap,&(CMAP->cregs),size);
}

void DoBODY(struct Animation *Anim,struct Frame *frame,struct BODY *BODY,
            struct ChunkHeader *CH)
{
   frame->Data.BODY=BODY;
   frame->DataSize=CH->size;

   CH->size=0;
}

void DoDPAN(struct Animation *Anim,struct Frame *Frame,struct DPAN *DPAN,
            struct ChunkHeader *CH)
{
   Anim->frames=DPAN->nframes;
}

void DoANHD(struct Animation *Anim,struct Frame *Frame,struct ANHD *ANHD,
            struct ChunkHeader *CH)
{
   if (ANHD->operation!=5) { IFFErr=IFF_BadAnimCompression; return; }
   if (ANHD->bits & ~2) { IFFErr=IFF_BadAnimModes; return; }

   if (!ANHD->interleave) ANHD->interleave=2;
   if (Anim->interleave)
   {
      if (Anim->interleave!=ANHD->interleave)
      { IFFErr=IFF_InterleaveChanged; return; }
   }
   else
   {
      if (ANHD->interleave>2) { IFFErr=IFF_BadInterleave; return; }
      Anim->interleave=ANHD->interleave;
   }

   Frame->reltime=ANHD->reltime;
   Frame->bits=ANHD->bits;
}

void DoInitANHD(struct Animation *Anim,struct Frame *Frame,struct ANHD *ANHD,
               struct ChunkHeader *CH)
{
   Frame->reltime=ANHD->reltime;
}

void DoDLTA(struct Animation *Anim,struct Frame *Frame,struct DLTA *DLTA,
            struct ChunkHeader *CH)
{
   UBYTE **DLTApointers;
   ULONG DLTABase;
   char i;

   Frame->Data.DLTA=DLTA;
   Frame->DataSize=CH->size;

   CH->size=0;

   DLTABase=(ULONG)DLTA+offsetof(struct DLTA,deltas);

   i=16; DLTApointers=DLTA->deltas;
   while(i--)

   {
   if (*DLTApointers) *DLTApointers+=DLTABase;
   DLTApointers++;
   }
}

struct BitMap *PrepareBitMap(struct BitMap *B,char depth,short width,
                              short height,struct BitMap *FBM)
{
   struct BitMap *b;
   unsigned char d;

   if (SystemVer>=39)
   {
      b=(struct BitMap *)AllocBitMap(width,height,depth,
                                    BMF_CLEAR | BMF_DISPLAYABLE,FBM);
   }
   else
   {
      b=B;
      InitBitMap(B,depth,width,height);
      for (d=0; d<depth; d++)
         if (!(B->Planes[d]=AllocRaster(width,height))) return(NULL);
         else BltClear(B->Planes[d],RASSIZE(width,height),0);
   }
   return(b);
}

void UnPrepareBitMap(struct BitMap *B,unsigned char depth,unsigned short width,
                     unsigned short height)
{
   unsigned char d;

   if (SystemVer>=39) FreeBitMap(B);
   else
   {
      for (d=0; d<depth; d++)
         if (B->Planes[d]) FreeRaster(B->Planes[d],width,height);
      else return;
   }
}

void FreeFrame(struct Frame *frame)
{
   if (frame->ColorMap) FreeColorMap(frame->ColorMap);

   if (frame->Data.BODY) FreeMem(frame->Data.BODY,frame->DataSize);


   FreeMem(frame,sizeof(struct Frame));
}

void FreeAnimation(struct Animation *Anim)
{
   struct Frame *frame,*nframe;
   struct ViewPort *vp;

   if (!Anim) return;

   if (Anim->Screen)
   {
      ScreenToBack(Anim->Screen);

      if (Anim->Window) CloseWindow(Anim->Window);

      Anim->Screen->ViewPort.ColorMap->ColorTable=Anim->ScreenColorMapData;
      Anim->Screen->ViewPort.ColorMap->Count=Anim->ScreenColorCount;
      if (Anim->CM39)
         Anim->Screen->ViewPort.ColorMap->TransparencyBits=
                                          Anim->ScreenColorMapLowData;
      if (SystemVer<36)
      {
         vp=&(Anim->Screen->ViewPort);

         Forbid();
         vp->DxOffset=Anim->OldDxOffset; vp->DyOffset=Anim->OldDyOffset;
         Anim->Screen->LeftEdge=Anim->OldSLeft;
         Anim->Screen->TopEdge=Anim->OldSTop;
         MakeScreen(Anim->Screen); RethinkDisplay();
         ScreenToFront(Anim->Screen);
      }

      CloseScreen(Anim->Screen);

      if (SystemVer<36) Permit();
   }

   nframe=Anim->FirstFrame;
   while(frame=nframe) { nframe=frame->NextFrame; FreeFrame(frame); }

   frame=&(Anim->InitialFrame);

   if (frame->ColorMap) FreeColorMap(frame->ColorMap);

   if (Anim->rBitMap)
      UnPrepareBitMap(Anim->rBitMap,Anim->depth,Anim->bm_width,Anim->bm_height);
   if (Anim->rAnimMap)
      UnPrepareBitMap(Anim->rAnimMap,Anim->depth,
                     Anim->bm_width,Anim->bm_height);

   FreeMem(Anim,sizeof(struct Animation));
}


struct Picture *LoadAnimation(char *fname)
{
   BPTR infile;
   struct IFFHeader IFFH;
   struct ChunkHeader CH;
   BOOL isANIM,isInitialFrame=TRUE;
   BOOL BMHD=FALSE,CAMG=FALSE,CMAP=FALSE,BODY=FALSE,ANHD=FALSE,DPAN=FALSE,
         DLTA=FALSE;
   ULONG Bytes,LBytes,IFFLevel=0,frames=0,ChunkSize;
   char pad;
   void *CD;
   struct Animation *Anim=NULL;
   struct Frame **FramePointer,*Frame;
   void (*ChunkHandler)(struct Animation *Anim,struct Frame *Frame,
                        void *ChunkBody,struct ChunkHeader *CH);

   if (!(infile=Open(fname,MODE_OLDFILE))) { IFFErr=IFF_NoFile; return(NULL); }

   if (Read(infile,&IFFH,sizeof(IFFH))!=sizeof(IFFH))
      { Close(infile); IFFErr=IFF_FileCorrupt; return(NULL); }

/* printfIFFLevel(IFFLevel++); printfFORM(&IFFH); printf("\n");*/

   if ((IFFH.group!=idFORM) || ((IFFH.type!=idILBM) && (IFFH.type!=idANIM)))
      { Close(infile); IFFErr=IFF_NoILBM_ANIM; return(NULL); }

   if (!(Anim=AllocMem(sizeof(struct Animation),MEMF_PUBLIC | MEMF_CLEAR)))
      { Close(infile); IFFErr=IFF_NoMemory; return(NULL); }

   isANIM=(IFFH.type==idANIM);
   Anim->flags=isANIM ? PIF_ANIM : 0;

   FramePointer=&(Anim->FirstFrame); Frame=&(Anim->InitialFrame);

   Bytes=IFFH.size-4;

   while(IFFErr==IFF_NoError)
   {
      if (isANIM)
      {
         if (Bytes<sizeof(struct IFFHeader)) { IFFErr=IFF_FileCorrupt; break; }
         Bytes-=sizeof(struct IFFHeader);
         if (Read(infile,&IFFH,sizeof(IFFH))!=sizeof(IFFH))
            { IFFErr=IFF_FileCorrupt; break; }
/*       printfIFFLevel(IFFLevel++); printfFORM(&IFFH); printf("\n");*/
         if ((IFFH.group!=idFORM) || (IFFH.type!=idILBM))
            { IFFErr=IFF_NoILBM_ANIM; break; }
         LBytes=IFFH.size-4;
         if (Bytes<LBytes)
            { IFFErr=IFF_FileCorrupt; break; }
      }
      else LBytes=Bytes;

      if (!isInitialFrame)
      {
         if (!(Frame=AllocMem(sizeof(struct Frame),
                                       MEMF_PUBLIC | MEMF_CLEAR)))
            { IFFErr=IFF_NoMemory; break; }

         *FramePointer=Frame;
         FramePointer=&(Frame->NextFrame);
      }

      Bytes-=LBytes;

      while(IFFErr==IFF_NoError)
      {
         if (LBytes<sizeof(struct ChunkHeader))
            { IFFErr=IFF_FileCorrupt; break; }
         LBytes-=sizeof(struct ChunkHeader);

         if (Read(infile,&CH,sizeof(CH))!=sizeof(CH))
            { IFFErr=IFF_FileCorrupt; break; }

         if (LBytes<CH.size) { IFFErr=IFF_FileCorrupt; break; }
         LBytes-=CH.size;

/*       printfIFFLevel(IFFLevel); printfChunkHeader(&CH); printf("\n");*/

         ChunkHandler=NULL;

         switch(CH.ID)
         {
            case idBMHD:
               if (BMHD) IFFErr=IFF_DoubleBMHD;
               BMHD=TRUE; ChunkHandler=DoBMHD;
               break;
            case idCMAP:
               if (CMAP) IFFErr=IFF_DoubleCMAP;
               CMAP=TRUE; ChunkHandler=DoCMAP;
               break;
            case idCAMG:
               if (CAMG) IFFErr=IFF_DoubleCAMG;
               CAMG=TRUE; ChunkHandler=DoCAMG;
               break;
            case idBODY:
               if (BODY) IFFErr=IFF_DoubleBODY;
               BODY=TRUE; ChunkHandler=DoBODY;
               break;
            case idDPAN:
               if (DPAN) IFFErr=IFF_DoubleDPAN;
               DPAN=TRUE; ChunkHandler=DoDPAN;
               break;
            case idANHD:
               if (ANHD) IFFErr=IFF_DoubleANHD;
               ANHD=TRUE;
               if (isInitialFrame) ChunkHandler=DoInitANHD;
               else ChunkHandler=DoANHD;
               break;
            case idDLTA:
               if (DLTA) IFFErr=IFF_DoubleDLTA;
               DLTA=TRUE; ChunkHandler=DoDLTA;
               break;
         }

         ChunkSize=CH.size;

         if (ChunkHandler)
         {

            if (CH.size)
            {
               if (!(CD=AllocMem(CH.size,MEMF_PUBLIC)))
                  { IFFErr=IFF_NoMemory; break; }
            }

            if (Read(infile,CD,CH.size)!=CH.size)
               { IFFErr=IFF_FileCorrupt; if (CD) FreeMem(CD,CH.size); break; }

            (*ChunkHandler)(Anim,Frame,CD,&CH);

            if (CH.size && CD) FreeMem(CD,CH.size);
         }
         else Seek(infile,CH.size,OFFSET_CURRENT);

         if (ChunkSize & 1)
         {
            if (!LBytes) { IFFErr=IFF_FileCorrupt; break; }
            LBytes--;
            if (Read(infile,&pad,1)!=1) { IFFErr=IFF_FileCorrupt; break; }
         }

         if (IFFErr!=IFF_NoError) break;

         if (!LBytes) IFFErr=IFF_End;
      }
      IFFLevel--;

      if (IFFErr!=IFF_End) break;
      if (Bytes) IFFErr=IFF_NoError;

      if (isInitialFrame)
      {
         if (!BMHD) { IFFErr=IFF_NoBMHD; break; }
         if (!CMAP) { IFFErr=IFF_NoCMAP; break; }
/*       if (!CAMG) { IFFErr=IFF_NoCAMG; break; } */
         if (CAMG) Anim->flags|=PIF_CAMG;
         if (!BODY) { IFFErr=IFF_NoBODY; break; }
         if (DLTA) { IFFErr=IFF_BadDLTA; break; }
         BMHD=CMAP=CAMG=BODY=ANHD=DPAN=FALSE;
         isInitialFrame=FALSE;

         if (!(Anim->rBitMap=PrepareBitMap(&(Anim->BitMap),Anim->depth,
                                             Anim->bm_width,Anim->bm_height,
                                             NULL)))
            { IFFErr=IFF_NoMemory; break; }

         if (Anim->InitialFrame.Data.BODY)
         {
            FillBitMap(Anim->rBitMap,Anim->width,Anim->height,Anim->depth,
                        Anim->masking==mskHasMask,NULL,
                        Anim->compression,Anim->InitialFrame.Data.BODY);
            if ((IFFErr!=IFF_End) && (IFFErr!=IFF_NoError)) break;
            FreeMem(Anim->InitialFrame.Data.BODY,Anim->InitialFrame.DataSize);
            Anim->InitialFrame.Data.BODY=NULL;
         }
      }
      else
      {
         if (BMHD) { IFFErr=IFF_BadBMHD; break; }
         if (CAMG) { IFFErr=IFF_BadCAMG; break; }
         if (BODY) { IFFErr=IFF_BadBODY; break; }
         if (DPAN) { IFFErr=IFF_BadDPAN; break; }
         if (!ANHD) { IFFErr=IFF_NoANHD; break; }
         if (!DLTA) { IFFErr=IFF_NoDLTA; break; }
         ANHD=CMAP=DLTA=FALSE;
         frames++;
      }
   }

   Anim->flags |= PIF_LOOP;

   if (IFFErr!=IFF_End) { FreeAnimation(Anim); Anim=NULL; }

   Close(infile);

   return(Anim);
}

long TestOverscan(USHORT w,USHORT h,ULONG mode,struct Rectangle *oclip,
               ULONG otype)
{
   SHORT ow,oh;

   if (!QueryOverscan(mode,oclip,otype)) return(-1);

   ow=oclip->MaxX-oclip->MinX+1; oh=oclip->MaxY-oclip->MinY+1;

   if ((w>ow) || (h>oh)) return(0);

   return(1);
}

ULONG CompatibleMode(ULONG mode)
{
   return(mode & (HIRES | LACE | HAM | EXTRA_HALFBRITE));
}

BOOL ReAllocBitMap(struct Animation *Anim,
                  unsigned short nbm_w,unsigned short nbm_h)
{
   struct BitMap *nbmp,nbm;

   if ((nbm_w!=Anim->bm_width) || (nbm_h!=Anim->bm_height))
   {
      if (!(nbmp=PrepareBitMap(&nbm,Anim->depth,
                              nbm_w,nbm_h,NULL)))
         return(FALSE);

      BltBitMap(Anim->rBitMap,0,0,nbmp,0,0,Anim->width,Anim->height,
               0xc0,0xff,NULL);
      WaitBlit();

      UnPrepareBitMap(Anim->rBitMap,Anim->depth,
                      Anim->bm_width,Anim->bm_height);

      if (nbmp==(&nbm))
      {
         CopyMem(nbmp,&(Anim->BitMap),sizeof(struct BitMap));
         Anim->rBitMap=&(Anim->BitMap);
      }
      else Anim->rBitMap=nbmp;

      Anim->bm_width=nbm_w; Anim->bm_height=nbm_h;
   }
   return(TRUE);
}

BOOL CreateScreen(struct Animation *Anim)
{
   ULONG mode,ctype=OSCAN_VIDEO+1,otype;
   long clipS;
   BOOL UseTags;
   long error=FALSE;
   struct Rectangle cclip,oclip;
   long ScrLeft,ScrTop,PlusLeft,MinusLeft;
   struct NewScreen NS;
   struct NewWindow NW;
   DisplayInfoHandle DispHandle;
   union
   {
      struct DisplayInfo DispInfo;
      struct DimensionInfo DispDimInfo;
   } DInfo;
   struct ColorMap *CMS,*CMF;
   struct BitMap HelpMap,*rHelpMap=NULL;
   WORD NDR,NDC,x,y;
   struct ViewPort *vp;
   unsigned short nbm_w,nbm_h;

   if (Anim->flags & PIF_CAMG) mode=Anim->ViewModes;
   else mode=(MONITOR_ID_MASK & ~EXTENDED_MODE);

   if ((!(mode & MONITOR_ID_MASK) ||
      ((mode & MONITOR_ID_MASK)==EXTENDED_MODE)))
   {
      mode&=(~(EXTENDED_MODE | SPRITES | GENLOCK_AUDIO | GENLOCK_VIDEO |
               VP_HIDE));
   }

   if ((mode & MONITOR_ID_MASK) && (!(mode & EXTENDED_MODE)))
   {
      mode=NULL;
      if (Anim->width >= 380) mode=HIRES;
      if (Anim->height >= 290) mode=LACE;
   }

   if (UseTags=(SystemVer>=36))
   {
      if (error=ModeNotAvailable(mode))
      {
         mode=CompatibleMode(mode);
         if (error=ModeNotAvailable(mode))
         {
            if (Anim->depth>6) { IFFErr=IFF_BadDims; return(FALSE); }
            if (Anim->depth>4) mode&=~HIRES;
            if ((mode & HAM) || (mode & EXTRA_HALFBRITE)) mode&=~HIRES;

            error=ModeNotAvailable(mode);
         }
      }
   }

   if (error) { IFFErr=IFF_UnknownScreenMode; return(FALSE); }

   if (UseTags)
   {
      nbm_w=Anim->bm_width;
      nbm_h=Anim->bm_height;

      if (!(DispHandle=FindDisplayInfo(mode))) UseTags=FALSE;
      else
      {
         if ((GetDisplayInfoData(DispHandle,(UBYTE *)&(DInfo.DispDimInfo),
                              sizeof(struct DimensionInfo),DTAG_DIMS,mode))<=0)
            UseTags=FALSE;
         else
         {
            if (DInfo.DispDimInfo.MaxDepth<Anim->depth) UseTags=FALSE;
            else
            if (DInfo.DispDimInfo.MaxRasterWidth<Anim->width) UseTags=FALSE;
            else
            if (DInfo.DispDimInfo.MaxRasterHeight<Anim->height) UseTags=FALSE;
            else
            {
               if (DInfo.DispDimInfo.MinRasterWidth>Anim->width)
                  nbm_w=DInfo.DispDimInfo.MinRasterWidth;
               if (DInfo.DispDimInfo.MinRasterHeight>Anim->height)
                  nbm_h=DInfo.DispDimInfo.MinRasterHeight;
               if (!ReAllocBitMap(Anim,max(nbm_w,64),max(nbm_h,64)))
               { IFFErr=IFF_BadDims; return(FALSE); }
            }
         }
      }
   }

   if (UseTags)
   {
      for (otype=OSCAN_TEXT;otype<=OSCAN_VIDEO;otype++)
      {
         clipS=TestOverscan(Anim->bm_width,Anim->bm_height,mode,&oclip,
                           OSCAN_TEXT);
         if (clipS>=0)
         {
            CopyMem(&oclip,&cclip,sizeof(struct Rectangle));
            ctype=otype;
            if (clipS) otype=OSCAN_VIDEO+1;
         }
      }
      if (ctype<=OSCAN_VIDEO)
      {
         ScrLeft=cclip.MinX+
                  ((cclip.MaxX-cclip.MinX+1-(short)Anim->bm_width)>>1);
         ScrTop=cclip.MinY+
                  ((cclip.MaxY-cclip.MinY+1-(short)Anim->bm_height)>>1);
         if (ScrTop>cclip.MinY) ScrTop=cclip.MinY;

         if (SystemVer>=39)
         {
            if (ScrLeft & 0x3F) PlusLeft=(ScrLeft & ~0x3F) + 0x40;
            MinusLeft=PlusLeft-0x80;
            if ((ScrLeft-MinusLeft)<(PlusLeft-ScrLeft))
               ScrLeft=MinusLeft;
            else
               ScrLeft=PlusLeft;
         }
      }
      else UseTags=FALSE;
   }

   if (UseTags)
      Anim->Screen=OpenScreenTags(NULL,
                                 SA_Top,ScrTop,
                                 SA_Left,ScrLeft,
                                 SA_Width,Anim->bm_width,
                                 SA_Height,Anim->bm_height,
                                 SA_Depth,Anim->depth,
                                 SA_ShowTitle,FALSE,
                                 SA_Quiet,TRUE,
                                 SA_BitMap,Anim->rBitMap,
                                 SA_Type,CUSTOMSCREEN,
                                 SA_Behind,TRUE,
                                 SA_DisplayID,mode,
                                 SA_Overscan,ctype,
                                 SA_AutoScroll,TRUE,
                                 TAG_END);

   if (!Anim->Screen)
   {
      if (!ReAllocBitMap(Anim,max(Anim->bm_width,64),max(Anim->bm_height,64)))
      { IFFErr=IFF_BadDims; return(FALSE); }

      mode=CompatibleMode(mode);

      NS.LeftEdge=NS.TopEdge=0;
      NS.Width=Anim->bm_width; NS.Height=Anim->bm_height;
      NS.Depth=Anim->depth;
      NS.DetailPen=0; NS.BlockPen=1;
      NS.ViewModes=mode;
      NS.Type=CUSTOMSCREEN | SCREENQUIET | CUSTOMBITMAP | SCREENBEHIND;
      NS.Font=NULL;
      NS.DefaultTitle=NULL;
      NS.Gadgets=NULL;
      NS.CustomBitMap=Anim->rBitMap;

      if (SystemVer<36)
      {
         if ((Anim->bm_width>1024) || (Anim->bm_height>1024))
            { IFFErr=IFF_BadDims; return(FALSE); }
         if (Anim->depth>6) { IFFErr=IFF_BadDims; return(FALSE); }
         if (Anim->depth>4) mode&=~HIRES;
         if ((mode & HAM) || (mode & EXTRA_HALFBRITE)) mode&=~HIRES;

         NS.ViewModes=mode;

         if (mode & HIRES) { if (NS.Width>736) NS.Width=736; }
         else { if (NS.Width>368) NS.Width=368; }

         if (mode & LACE) { if (NS.Height>580) NS.Height=580; }
         else { if (NS.Height>290) NS.Height=290; }

         rHelpMap=PrepareBitMap(&HelpMap,Anim->depth,NS.Width,12,NULL);
         if (rHelpMap)
            BltBitMap(Anim->rBitMap,0,0,rHelpMap,0,0,NS.Width,12,0xc0,0xff,
                     NULL);
         else
            NS.Type&=~SCREENQUIET;
      }

      Anim->Screen=OpenScreen(&NS);
   }

   if (!Anim->Screen)
   {
      if (rHelpMap) UnPrepareBitMap(rHelpMap,Anim->depth,NS.Width,12);
      IFFErr=IFF_NoScreen;
      return(FALSE);
   }

   if (SystemVer<36)
   {
      NDR=GfxBase->NormalDisplayRows; NDC=GfxBase->NormalDisplayColumns;
      x=Anim->Screen->Width; y=Anim->Screen->Height;
      if (NDR>300) NDR>>=1; if (NDC>400) NDC>>=1;
      vp=&(Anim->Screen->ViewPort);
      x-=NDC; if (vp->Modes & HIRES) x-=NDC;
      y-=NDR; if (vp->Modes & LACE) y-=NDR;

      x>>=1; y>>=1; if (y<0) y=0;
      Forbid();
      ScreenToFront(Anim->Screen);
      if (vp->Modes & HAM)
         if (GfxBase->ActiView->DxOffset-x<96)
            x=GfxBase->ActiView->DxOffset-96;

      Anim->OldDxOffset=vp->DxOffset;
      Anim->OldDyOffset=vp->DyOffset;
      Anim->OldSLeft=Anim->Screen->LeftEdge;
      Anim->OldSTop=Anim->Screen->TopEdge;

      vp->DxOffset=-x; vp->DyOffset=-y;
      Anim->Screen->LeftEdge=-x; Anim->Screen->TopEdge=-y;
      MakeScreen(Anim->Screen); RethinkDisplay();
      Permit();
   }

   if (rHelpMap)
   {
      BltBitMap(rHelpMap,0,0,Anim->rBitMap,0,0,NS.Width,12,0xc0,0xff,NULL);
      UnPrepareBitMap(rHelpMap,Anim->depth,NS.Width,12);
   }

   Forbid();
   CMS=Anim->Screen->ViewPort.ColorMap;

   Anim->CM39=(CMS->Type>=/*COLORMAP_TYPE_V39*/2);
   Anim->ScreenColorMapData=CMS->ColorTable;
   Anim->ScreenColorCount=CMS->Count;
   if (Anim->CM39) Anim->ScreenColorMapLowData=CMS->TransparencyBits;

   if (CMF=Anim->InitialFrame.ColorMap)
   {
      CMS->ColorTable=CMF->ColorTable;
      CMS->Count=CMF->Count;
      if (Anim->CM39) CMS->TransparencyBits=CMF->TransparencyBits;
   }

   MakeScreen(Anim->Screen); RethinkDisplay();
   Permit();

   NW.LeftEdge=NW.TopEdge=0;
   NW.Width=Anim->Screen->Width; NW.Height=Anim->Screen->Height;
   NW.DetailPen=0; NW.BlockPen=1;
   NW.IDCMPFlags=0;
   NW.Flags=SUPER_BITMAP | BACKDROP | BORDERLESS | RMBTRAP;
   NW.FirstGadget=NULL;
   NW.CheckMark=NULL;
   NW.Title=NULL;
   NW.Screen=Anim->Screen;
   NW.BitMap=Anim->rBitMap;
   NW.MinWidth=NW.MinHeight=NW.MaxWidth=NW.MaxHeight=0;
   NW.Type=CUSTOMSCREEN;

   if (SystemVer<36) ShowTitle(Anim->Screen,FALSE);

   if (!(Anim->Window=OpenWindow(&NW))) { IFFErr=IFF_NoScreen; return(FALSE); }

   return(TRUE);
}

void StartMessages(struct Window *W,ULONG iflags)
{ ModifyIDCMP(W,iflags); }

void WaitMessages(struct Window *W)
{ WaitPort(W->UserPort); }

BOOL DoMessages(struct Window *W)
{
   struct IntuiMessage *IM;
   BOOL cnt=TRUE;

   while(cnt && (IM=(struct IntuiMessage *)GetMsg(W->UserPort)))
   {
      CopyMem(IM,&IMCopy,sizeof(struct IntuiMessage));
      ReplyMsg((struct Message *)IM);

      cnt=TestMessage(&IMCopy);
   }

   return(cnt);
}

void StopMessages(struct Window *W)
{
   struct MsgPort *WP;
   struct IntuiMessage *IM;

   Forbid();
   WP=W->UserPort; W->UserPort=NULL;
   ModifyIDCMP(W,NULL);
   Permit();

   while(IM=(struct IntuiMessage *)GetMsg(WP)) ReplyMsg((struct Message *)IM);
   DeletePort(WP);
}

void UpDateScreen(struct Screen *scr)
{
   MakeScreen(scr); RethinkDisplay();
}

void MyAnimateAnim5(struct Animation *Anim)
{
   struct BitMap **Primary,**Secondary,*BM;
   struct ColorMap *PrimaryCM,*SecondaryCM,*CM;
   struct Frame *Frame;
   struct ColorMap *SCMap;
   BOOL cnt=TRUE;
   BOOL swapped=FALSE;
   ULONG LineBytes=((Anim->width+15)>>4)<<1;

/* long sSecs,sMicros,eSecs,eMicros,Secs,Micros,fSecs,fMicros;*/
/* ULONG loops=0,frames=0;*/

/* CurrentTime(&sSecs,&sMicros);*/

   Primary=&((ViewPortAddress(Anim->Window))->RasInfo->BitMap);
   PrimaryCM=Anim->InitialFrame.ColorMap;
   SCMap=(ViewPortAddress(Anim->Window))->ColorMap;

   Secondary=&(Anim->rAnimMap); SecondaryCM=PrimaryCM;

   if (!(*Secondary=PrepareBitMap(&(Anim->AnimMap),Anim->depth,
                                    Anim->bm_width,Anim->bm_height,
                                    *Primary)))
         return;

   BltBitMap(*Primary,0,0,*Secondary,0,0,Anim->width,Anim->height,
            0xc0,0xff,NULL);

   Frame=Anim->FirstFrame;

   while(cnt)
   {
      if (Anim->interleave==2)
      {
         BM=*Primary; *Primary=*Secondary; *Secondary=BM;
         CM=PrimaryCM; PrimaryCM=SecondaryCM; SecondaryCM=CM;
         swapped=!swapped;
      }
      else
      {
         BltBitMap(*Secondary,0,0,*Primary,0,0,Anim->width,Anim->height,
                  0xc0,0xff,NULL);
         PrimaryCM=SecondaryCM;
      }

      Forbid();
      SCMap->ColorTable=PrimaryCM->ColorTable;
      SCMap->Count=PrimaryCM->Count;
      if (Anim->CM39) SCMap->TransparencyBits=PrimaryCM->TransparencyBits;

      UpDateScreen(Anim->Screen);
      Permit();

      if (Frame)
      {
         USE_FRAME(*Secondary,Frame,LineBytes);
         if (Frame->ColorMap) SecondaryCM=Frame->ColorMap;
      }

      Frame=Frame->NextFrame; /*frames++;*/
      if (!Frame)
      {
         if (Anim->flags & PIF_LOOP)
         {
/*          loops++;*/
            Frame=Anim->FirstFrame;
            if (Anim->interleave==2) Frame=Frame->NextFrame;
         }
         else cnt=FALSE;
      }
      cnt=DoMessages(Anim->Window);
   }

/* CurrentTime(&eSecs,&eMicros);

   Secs=eSecs-sSecs; Micros=eMicros-sMicros;
   if (Micros<0) { Micros=1000000+Micros; Secs=Secs-1; }

   printf("Time: %ld secs, %ld micros\n",Secs,Micros);
   printf("Loops: %ld      Frames: %ld\n",loops,frames);

   fSecs=Secs/frames; fMicros=(Micros+1000000*(Secs-fSecs*frames))/frames;

   printf("\nTime per frame: %ld secs, %ld micros\n",fSecs,fMicros);*/

   if (swapped)
   {
      Forbid();
      BM=*Primary; *Primary=*Secondary; *Secondary=BM;
      UpDateScreen(Anim->Screen);
      Permit();
   }
}

ULONG ShowIFF(char *IFFName,ULONG IDCMPFlags,ULONG MyFlags,ULONG UserMode)
{
   struct Animation *Anim;

   IFFErr=IFF_NoError;

   Anim=LoadAnimation(IFFName);

   if (Anim)
   {
      if (MyFlags & USER_MODE)
      {
         if (Anim->flags & PIF_CAMG)
         {
            if (Anim->ViewModes & HAM) UserMode|=HAM;
            if (Anim->ViewModes & EHB) UserMode|=EHB;
         }
         Anim->ViewModes=UserMode; Anim->flags|=PIF_CAMG;
      }

      if (CreateScreen(Anim))
      {
         if (MyFlags & CLEAR_POINTER)
         {
            GetNoPointer();
            SetPointer(Anim->Window,NoPointer,0,0,0,0);
         }

         TestMessage(NULL);

         ScreenToFront(Anim->Screen); ActivateWindow(Anim->Window);

         StartMessages(Anim->Window,IDCMPFlags);

         if (Anim->flags & PIF_ANIM)
         {
            MyAnimateAnim5(Anim);
         }
         else
         {
            do
               WaitMessages(Anim->Window);
            while(DoMessages(Anim->Window));
         }

         StopMessages(Anim->Window);

         if (MyFlags & CLEAR_POINTER)
         {
            ClearPointer(Anim->Window);
            UnGetNoPointer();
         }
      }
      FreeAnimation(Anim);
   }

   return((IFFErr==IFF_End) ? 0 : IFFErr);
}
