/*
 * Example user-space parport driver
 *
 * Copyright (c) 1998 Tim Waugh <tim@cyberelk.demon.co.uk>
 */

#include <time.h>
#include <stdio.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <linux/parport.h>
#include "ppuser.h"

static void write_parport (int p, const char * str)
{
	const struct timespec w = {0, 1000};
	unsigned int ctr = (PARPORT_CONTROL_SELECT |
			    PARPORT_CONTROL_INIT);

	/* Claim the port */
	if (ioctl (p, PPCLAIM, 0)) {
		printf ("Couldn't claim port\n");
		return;
	}

	for (; *str; str++) {
		/* Write character to data register */
		if (ioctl (p, PPWDATA, str)) {
			printf ("Couldn't write data\n");
			return;
 		}

		/* Wait a tiny bit */
		nanosleep (&w, NULL);

		/* Fiddle with control register */
		ctr |= PARPORT_CONTROL_STROBE;
		if (ioctl (p, PPWCONTROL, &ctr)) {
			printf ("Couldn't write control\n");
			return;
		}

		/* Wait another tiny bit */
		nanosleep (&w, NULL);

		/* Fiddle with control register again */
		ctr &= ~PARPORT_CONTROL_STROBE;
		if (ioctl (p, PPWCONTROL, &ctr)) {
			printf ("Couldn't write control\n");
			return;
		}

		/* Wait another tiny bit */
		nanosleep (&w, NULL);
	}
	
	/* Release the port */
	ioctl (p, PPRELEASE, 0);
}
	
/* Greet parports */
int main (int argc, char * argv [])
{
	int n;

	for (n = 0; ; n++) {
		char name[16];
		int p;
		sprintf (name, "/dev/ppuser%d0", n);
		p = open (name, O_RDWR);
		if (p == -1) {
			fprintf (stderr, "couldn't open %s:  ", name);
			perror("");
			if (n == 0)
				printf ("Did you make the /dev/ppuser00 device?\n"
					"Did you insmod ppuser.o?\n");
			break;
		}

		printf ("Attempting to greet %s\n", name);
		write_parport (p, "hello world\f");
		close (p);
	}

	return n;
}
