NAME 	Loadcode
codeseg	segment 'code'

;-----------------------------------------------------------------------------;
; This is an example assembly language program designed to show how to        ;
; use the DOS load overlay function to load a portion of code in expanded     ;
; memory and then how to use EMM to call it.                                  ;
;                                                                             ;
; The major steps it follows are:                                             ;
;       Test for presence of EMM                                              ;
;       Allocate a page of expanded memory                                    ;
;       Map the page in                                                       ;
;       Use DOS load overlay to load the program HELLO.EXE to the mapped page ;
;       Use EMS map_and_call to start execution of HELLO.EXE                  ;
;       Deallocate the page                                                   ;
;       Terminate                                                             ;
;                                                                             ;
; The "overlay" code that gets executed out of expanded memory MUST follow    ;
; strict guidelines:                                                          ;
;       The Stack MUST be left in conventional memory                         ;
;       A "RETF" far return should end the code so that the kernal in         ;
;               conventional memory can deallocate the EMS memory             ;
;                                                                             ;
; Executing code out of expanded memory is best done in assembly language.    ;
; High level languages (even C) don't offer enough control of where the       ;
; stack is and how the "overlay" file gets built.  Note that even though      ;
; HELLO.EXE has been linked into an "EXE" file it is NOT actually executable  ;
; from the DOS prompt.                                                        ;
;-----------------------------------------------------------------------------;
ASSUME	CS:codeseg, DS:codeseg, ES:codeseg

;-----------------------------------------------------------------------------;
;          E Q U A T E S                                                      ;
;-----------------------------------------------------------------------------;
PAGE;
;======================================;
;	Equates used by the EMM        ;
;======================================;

	;-----------------------------------------------------;
	; define the enhanced memory manager access interrupt ;
	;-----------------------------------------------------;
EMM		EQU	67H		;interrupt to use to access EMM functions

	;-----------------------------------------;
	; define the first page in the page frame ;
	; nad the logical page we will use.       ;
	;-----------------------------------------;
phys_page_0	EQU	0
log_page_0	EQU	0

	;------------------------------------------;
	; define enhanced memory manager functions ;
	;------------------------------------------;
GetEMMStatus		EQU	40h	;function used to check for EMM present
GetPageFrame		EQU	41h	;function used to get the page frame address
AllocatePages		EQU	43h	;function used to get a pID and allocate pages
MapPage			EQU	44h	;function used to map a memory page into the page frame
DeallocatePages 	EQU	45h	;function used to deallocate pages

;======================================;
;	Equates used for DOS           ;
;======================================;
Load_overlay            EQU	4B03H	;function used to load overlay
DOS_print_string	EQU	09H	;DOS print function
DOSTerminate		EQU	4CH	;Dos terminate function


PAGE
;=======================================================================;
;									;
;  Procedure test_for_EMM						;
;									;
;  Description:  This procedure tests for the presence of EMM in the	;
;		 system and will return with the CARRY FLAG SET if the	;
;		 manager is present, otherwise it will return with the	;
;		 CARRY FLAG CLEAR.					;
;									;
;  Parameters:	 None							;
;									;
;  Calling Sequence:  CALL	test_for_EMM				;
;		      JCS	errorHandler				;
;									;
;  Register usage:  Carry flag set to indicate status returned		;
;									;
;=======================================================================;

test_for_EMM	PROC	NEAR

		PUSH	ES

		MOV	AH, 35h 		;issue "get interrupt vector"
		MOV	AL, 67h
		INT	21H

		MOV	DI, 000Ah		;use the SEGMENT in ES returned
						;  by DOS, place the "device
						;  name field"  OFFSET in DI
		LEA	SI, device_name 	;place the OFFSET of the
						;  EMMXXXX0 name string in SI,
						;  SEGMENT is already in DS
		MOV	CX, 8			;compare the name strings
		REPE	CMPSB

		JNE	test_for_EMM_error_exit ;the driver name string:

test_for_EMM_exit:
		STC				;was found, so EMM is present
		POP	ES
		RET

test_for_EMM_error_exit:
		CLC				;was not found, EMM is not present
		POP	ES
		RET

device_name:	DB	"EMMXXXX0"

test_for_EMM	ENDP

PAGE
;-----------------------------------------------------------------------------;
;                                                                             ;
;                                                                             ;
;-----------------------------------------------------------------------------;

loadcode_main	PROC	NEAR

	;---------------------------------------------------------------------;
	; Set DS, ES segments to our CS                                       ;
	;---------------------------------------------------------------------;
	MOV	AX, CS
	MOV	ES, AX
	MOV	DS, AX

	;---------------------------------------------------------------------;
	; Test for EMM's presence                                             ;
	;---------------------------------------------------------------------;
	CALL	test_for_EMM
	JNC	error			 	;No carry set - EMM not found

	;---------------------------------------------------------------------;
	; get segment address of the page frame from EMM                      ;
	;---------------------------------------------------------------------;
	MOV		AH, GetPageFrame	;function to get address of page frame
	INT		EMM
	OR		AH, AH			;Test for error return from EMM
	JNZ		error

	;---------------------------------------------------------------------;
	; save the segment address of the page frame                          ;
	;---------------------------------------------------------------------;
	MOV		page_frame_addr, BX

	;---------------------------------------------------------------------;
	; Allocate enough pages to load HELLO.EXE - 1 page should be enough   ;
	;---------------------------------------------------------------------;
	MOV		AH, AllocatePages	;function to get a pID and pages assigned
	MOV		BX, 1              	;number of pages to request
	INT		EMM
	OR		AH, AH			;Test for error return from EMM
	JNZ		error

	;---------------------------------------------------------------------;
	; save the handle returned by EMM                                     ;
	;---------------------------------------------------------------------;
	MOV		handle, DX 

	;---------------------------------------------------------------------;
	; map in the logical page at physical page 0                          ;
	;---------------------------------------------------------------------;
	MOV		DX, handle 		;set up process ID for EMM call
	MOV		BX, log_page_0		;map in our only logical page
	MOV		AL, phys_page_0		;designate page 0 in frame
	MOV		AH, MapPage		;function to map in page
	INT		EMM			;execute function
	OR		AH, AH			;Test for error return from EMM
	JNZ		error

	;---------------------------------------------------------------------;
	; Use DOS overlay function to load HELLO.EXE into expanded memory     ;
	;---------------------------------------------------------------------;
	PUSH		DS
	POP		ES
	LEA		BX, hello_overlay_info	;ES:BX --> overlay struct

	MOV		AX, page_frame_addr		;Set overlay info to the
	MOV		ES:[BX].segment_addr, AX   	;page frame segment
	MOV		ES:[BX].relocation_factor, AX

	MOV		DX, OFFSET hello_filename	;DS:DX --> ASCIIZ filename

	MOV		AX, Load_overlay		;Use DOS to load the overlay
	INT		21H
	JC		error

	;---------------------------------------------------------------------;
	; Set up hello's execution address and call hello                     ;
	;---------------------------------------------------------------------;
	MOV		AX, page_frame_addr
	MOV		hello_segment, AX

	LEA		BX, hello_execution_addr

	CALL 		DWORD PTR [BX]

terminate:	
	MOV		AH, DeallocatePages
	MOV		DX, handle
	INT		EMM
	OR		AH, AH
	JNZ		error

terminate_exit:
	MOV		AH, DosTerminate	;DOS terminate command
	MOV		AL, 0			;load normal return code
	INT		21h

PAGE
	;=====================================================================;
	; if there were EMM or DOS errors,                                    ;
	;     display error message                                           ;
	;=====================================================================;

error:
	LEA		DX, error_msg
	MOV		AH, DOS_print_string
	INT		21h
	JMP		terminate_exit

loadcode_main	ENDP

PAGE
;-----------------------------------------------------------------------------;
;									      ;
;		D A T A                       				      ;
;									      ;
;-----------------------------------------------------------------------------;

	;=====================================================================;
	;                        EMM related variables                        ;
	;=====================================================================;
	;---------------------------------------------------------------------;
	; handle returned by EMM.                                             ;
	;---------------------------------------------------------------------;
handle			DW	?	;process ID returned by EMM when pages
                                        ;  are allocated

	;---------------------------------------------------------------------;
	; address of the EMM page frame                                       ;
	;---------------------------------------------------------------------;
page_frame_addr		DW	?		;segment address of top of page frame

	;---------------------------------------------------------------------;
	; Structure passed to DOS to load an overlay                          ;
	;---------------------------------------------------------------------;
	Overlay_struc			STRUC
		segment_addr		DW (?)
		relocation_factor	DW (?)
	Overlay_struc			ENDS

	hello_overlay_info		Overlay_struc <>

	;---------------------------------------------------------------------;
	; Filename of overlay to load                                         ;
	;---------------------------------------------------------------------;
hello_filename	DB	'Hello.exe', 0

	;---------------------------------------------------------------------;
	; Address to begin execution of the overlay                           ;
	;---------------------------------------------------------------------;
hello_execution_addr	LABEL DWORD
	hello_offset	DW	100H 	;Depends on hello.exe
	hello_segment	DW	?	;Will be set to page_frame_addr

	;---------------------------------------------------------------------;
	; Messages to be displayed                                            ;
	;---------------------------------------------------------------------;
error_msg	DB 'Error encountered.', '$'

codeseg 	ENDS
		
END	loadcode_main

