/* ---------------------------------------------------------------------- */
/*   Stix.c (C) Copyright Bill Buckels 1989-1999                          */
/*   All Rights Reserved.                                                 */
/*                                                                        */
/*   Licence Agreement                                                    */
/*   -----------------                                                    */
/*                                                                        */
/*   STIX is distributed as ShareWare.                                    */
/*   Suggested Registration is $10.00 per family.                         */
/*                                                                        */
/*   You are expected to register with the Author if you use STIX beyond  */
/*   a 30-day evaluation period. Send registration in the form of         */
/*   cheque, or money order to:                                           */
/*                                                                        */
/*   Bill Buckels                                                         */
/*   589 Oxford Street                                                    */
/*   Winnipeg, Manitoba, Canada R3M 3J2                                   */
/*                                                                        */
/*   Email: bbuckels@escape.ca                                            */
/*   WebSite: http://www.escape.ca/~bbuckels                              */
/*                                                                        */
/*   Registered users of STIX have a royalty-free right to use, modify,   */
/*   reproduce and distribute this source code (and/or any modified       */
/*   version) in way you find useful, provided that you do not compete    */
/*   with Bill Buckels or his agents, and that you agree that Bill        */
/*   Buckels has no warranty obligations or liability whatsoever          */
/*   resulting from any associated loss or damage.                        */
/*                                                                        */
/*   If you do not agree with these terms, remove this source and         */
/*   all associated files from your computer now.                         */
/*                                                                        */
/*   Description                                                          */
/*   -----------                                                          */
/*                                                                        */
/*   STIX is a Children's Sticker Art System and Drawing Program. It      */
/*   allows the child to create "Sticker Art" by combining a library      */
/*   of 148 "Stickers" and  "Big Letters" onto a selection of             */
/*   "BackGrounds". (STIX comes with a variety of "Backgrounds" which     */
/*   can be used as a "Canvas" for the young artist's "MasterPiece".)     */
/*                                                                        */
/*   When your child is finished with the "MasterPiece", it can be        */
/*   safely saved away into "Your Library", and later showed-off to       */
/*   friends and family, or used as the background for yet another        */
/*   "MasterPiece".                                                       */
/*                                                                        */
/*   No "real" typing is required (unless the child wants to use the      */
/*   "ABC" Menu Command and add "Big Letters" and "Big Words" to the      */
/*   "Masterpiece").                                                      */
/*                                                                        */
/*   The child uses STIX with the mouse or the arrow keys and "F-Keys"    */
/*   on the keyboard.                                                     */
/*                                                                        */
/*   STIX is written in Large Model Microsoft C Version 6.00a             */
/*   STIX Uses 4-Color CGA .PCX files.                                    */
/*                                                                        */
/* ---------------------------------------------------------------------- */

#include <stdlib.h>
#include <stdio.h>
#include <fcntl.h>
#include <string.h>
#include <dos.h>
#include <bios.h>
#include <io.h>
#include <time.h>
#include <malloc.h>
#include <conio.h>
#include <graph.h>
#include <process.h>

/* ---------------------------------------------------------------------- */
/* Constants and Macros                                                   */
/* ---------------------------------------------------------------------- */
#define TRUE     1
#define FALSE    0

/* CGA screen modes */
#define TEXT     3
#define CGA_320  5
#define HI_RES   6

/* CGA Color Block constants for clearing the screen */
#define BLKBLK '\x00'
#define BLUBLK '\x55'
#define REDBLK '\xaa'
#define WHTBLK '\xff'

/* CGA palettes */
#define GRY     0
#define CMW     1

#define BLACK     0       /* standard IBM PC bios color indexes */
#define BLUE      1
#define GREEN     2
#define CYAN      3
#define RED       4
#define MAGENTA   5
#define BROWN     6
#define WHITE     7
#define GRAY      8
#define LBLUE     9
#define LGREEN    10
#define LCYAN     11
#define LRED      12
#define LMAGENTA  13
#define YELLOW    14
#define BWHITE    15
#define INTENSE   16      /* add to standard color to bump the intensity   */

#define ENTERKEY   '\x0d' /* character generated by the Enter Key          */
#define ESCKEY     '\x1b' /* character generated by the Esc key            */
#define FUNCKEY    '\x00' /* first character generated by function keys    */
#define UPARROW    'H'    /* second character generated by up-arrow key    */
#define DOWNARROW  'P'    /* second character generated by down-arrow key  */
#define LTARROW    'K'    /* second character generated by left-arrow key  */
#define RTARROW    'M'    /* second character generated by right-arrow key */
#define PGUP       'I'    /* second character generated by page up key     */
#define PGDOWN     'Q'    /* second character generated by page down key   */
#define HOMEKEY    'G'    /* second character generated by home key        */
#define ENDKEY     'O'    /* second character generated by end key         */

/* second character generated by numerical fkeys */
/* starting at character 59                      */
/* ending at character 68                        */
#define F1         ';'
#define F2         '<'
#define F3         '='
#define F4         '>'
#define F5         '?'
#define F6         '@'
#define F7         'A'
#define F8         'B'
#define F9         'C'
#define F10        'D'

#define CRETURN '\x0d'
#define LFEED   '\x0A'
#define CTRLZ   '\x1a'
#define DELETE  '\x7f'
#define ESCAPE  '\x1b'
#define BACKSPACE   8
#define SPACEBAR    32

#define SCREEN_ADDRESS (unsigned char *) 0xB8000000l
#define SCREENBUF_SIZE 16385
#define CLIPBUF_SIZE   4096
#define PATHLEN 13

#define FileExists(file) (_dos_findfirst(file,_A_NORMAL,&WildCard)==0)

#define VALID     0
#define INVALID  -1
#define INVALID_CHAR 255

#define XMOS 159
#define YMOS 99
#define XMAX 319
#define YMAX 199

enum {
  BLACK_CGA = 0,
  CYAN_CGA,
  RED_CGA,
  WHITE_CGA};

enum {
  HOTMAIN_MENU = 0,
  HOTCLIPS_MENU,
  GETTHEPICTURE_MENU};

/* a bit field structure for the CGA pixel */
struct cgabit{
       unsigned x1: 2;
       unsigned x2: 2;
       unsigned x3: 2;
       unsigned x4: 2;
    };

/* we use a union to unmask the CGA pixel bit fields */
union cgapixel{
      struct cgabit pixel;
      unsigned char byt;
    };

/* an array for a sorted file list */
typedef struct
{
  char name[PATHLEN];
}WILDFILES;

/* ---------------------------------------------------------------------- */
/* Global Vars, etc.                                                      */
/* ---------------------------------------------------------------------- */
int bMickeyMouse = FALSE,
    iMousex = XMOS,
    iMousey = YMOS;

WILDFILES *WildFiles = (WILDFILES *)NULL;
int iWildCounter = 0;
struct find_t WildCard;

FILE *pcxfile = NULL;
char szPcxFileName[66];
unsigned char szPcxHeader[128];

int iScreenMode=CGA_320,
    iDrawColor = WHITE_CGA,
    iShadowColor = CYAN_CGA,
    iMenuTag=HOTMAIN_MENU,
    iCurrentChoice=0;

int PicPosition[6][4]={
    0  , 0,  106, 83,
    107 ,0,  211, 83,
    212 ,0,  319, 83,

    0  ,117, 106, 199,
    107,117, 211, 199,
    212,117, 319, 199};


int StickBox[6][4]={
    4,  0, 107,95,
    108,0, 211,95,
    212,0, 315,95,
    4,  96,107,191,
    108,96,211,191,
    212,96,315,191};

int TitleBox[8][4]={
    44 ,84, 67 ,94,        /* f1 */
    148,84, 171,94,        /* f2 */
    252,84, 275,94,        /* f3 */
    44 ,180,67 ,190,       /* f4 */
    148,180,171,190,       /* f5 */
    252,180,275,190,       /* f6 */
    0  ,192,159,199,       /* enter */
    160,192,319,199};      /* pgup, pgdown*/

char *ClipText[8]={
    "F1","F2","F3","F4","F5","F6","Enter to Choose","PgDn - PgUp"};

char *lpClipBuffer,
     *lpScreenBuffer,
     *lpMenuBuffer,
     *lpWorkBuffer;

int iFirstMenu = FALSE;

short Arrow[16][8]={   /* the cursor */
3,2,3,0,0,3,2,3,       /* an easy method to design a tiny artifact */
2,3,2,0,0,2,3,2,       /* this one's called Thomas Q. Mouse...     */
3,2,3,0,0,3,2,3,
0,0,3,3,3,3,0,0,       /* this type of array is usually referred   */
0,3,3,3,3,3,3,0,       /* to as a sprite in GWBASIC.               */
3,3,0,3,3,0,3,3,
3,3,3,0,0,3,3,3,
3,2,3,3,3,3,2,3,
3,2,2,2,2,2,2,3,
0,3,3,3,3,3,3,0,
1,1,1,1,1,1,1,1,
3,3,1,1,1,1,3,3,
3,3,0,0,0,0,3,3,
0,1,0,1,0,1,0,0,
0,0,1,0,1,0,1,0,
3,3,3,3,3,3,3,3};

short Zorro[16][8]={   /* the screen mask */
0,0,0,3,3,0,0,0,       /* the outline of the shape defined by arrow */
0,0,0,3,3,0,0,0,       /* is anded with 3 to obtain an invisible    */
0,0,0,3,3,0,0,0,       /* appearance. other values are anded with 0 */
3,3,0,0,0,0,3,3,       /* and maintain their integrity              */
3,0,0,0,0,0,0,3,
0,0,0,0,0,0,0,0,
0,0,0,0,0,0,0,0,
0,0,0,0,0,0,0,0,
0,0,0,0,0,0,0,0,
3,0,0,0,0,0,0,3,
0,0,0,0,0,0,0,0,
0,0,0,0,0,0,0,0,
0,0,0,0,0,0,0,0,
3,0,0,0,0,0,0,3,
3,0,0,0,0,0,0,3,
0,0,0,0,0,0,0,0};

static int Cursor[32]={ /* some dummy values */
0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0};

int *CursorPtr=Cursor;
int iSoundFlag = TRUE;

/* ---------------------------------------------------------------------- */
/* Function Prototypes                                                    */
/* ---------------------------------------------------------------------- */

int HotMain (int choice);
int HotClips (void);
void HotPaste (void);
int HotLetters (void);
int GetThePicture (char *pszSearchSpec, char *pszSearchFilter);
int SaveThePicture (void);
int InitStick (void);
void DoneStick (void);
void TitleStuff (void);
int SetBuffers (void);
void FreeWildFiles (void);
int IsFormatValid (char *pszFileName, char *pszFormatSpec);
int GetWildFiles (char *pszSearchSpec, char *pszFormatSpec);
unsigned int byteword (unsigned char a, unsigned char b);
unsigned char lsb (unsigned int word);
unsigned char msb (unsigned int word);
int CheckForPcx (char *name);
int PcxLoad (unsigned char *crt);
int MakePcxHeader (void);
int SaveToPcx (char *pszOutFile);
int encline (unsigned char *inbuff, int inlen);
int encput (unsigned char byt, unsigned char cnt);
int compare();
int CheckForCGA (void);
void SetCrtMode (int m);
int SetScreenColor (unsigned background, unsigned palette);
void Zap (char zapcolor);
void PutDot (unsigned int col, unsigned int row, unsigned uiColor);
void LineBox (int x1, int y1, int x2, int y2, int linecolor);
void FilledBox (int x1, int y1, int x2, int y2, int iColor);
void PCRomFont (unsigned char *str,
	 int xorigin,
	 int yorigin,
	 int scale,
	 int fontcolor);
void PCMidFont (unsigned char *str,
	 int xmiddle,
	 int yorigin,
	 int scale,
	 int fontcolor);
void SaveScreen (char *ptr);
void RestoreScreen (char *ptr);
void MirrorScreen (char *ptr);
void GetClip (int x1, int y1, int x2, int y2);
void PutClip (int x1, int y1, int x2, int y2);
int ClipMove (int x1, int y1, int x2, int y2);
int sound (int freq, int tlen);
int KeyClick (void);
int Bronx (void);
int PicMouse (int x1, int y1);
int HotClick (int iRetVal, int *position, int *oldposition);
int ClickOnly (void);
int StickMouse (int x1, int y1);
int HotStick (int iRetVal, int *position, int *oldposition);
int StickClick (int iRetVal, int *xtest, int *ytest);
int HideMickey (void);
int ShowMickey (void);
int VerticalMickeyMinMax (int ymin, int ymax);
void MakeMickey (void);
int HiMickey (void);

/* ---------------------------------------------------------------------- */
/* Main Program                                                           */
/* ---------------------------------------------------------------------- */
void main(int argc,char *argv[])
{
  int iSavedChoice,
      status;

  iSavedChoice = iCurrentChoice = 0;
  InitStick();

  for (;;) {
    iMenuTag=HOTMAIN_MENU;
    iSavedChoice = iCurrentChoice = HotMain(iSavedChoice);
    if (iCurrentChoice != 2)
      KeyClick();
    switch(iCurrentChoice) {
      case 0:
          iMenuTag=HOTCLIPS_MENU;
          if (HotClips() == INVALID)
            break;
          HotPaste();
          break;           /*  STICKERS */
      case 1:
          HotLetters();
          break;           /*  LETTERS  */
      case 2:
          status = GetThePicture("save????.pcx", "save9999.pcx");
          if (status == INVALID)
            Bronx();
          else
            KeyClick();
          break;           /*  LIBRARY  */
      case 3:
          GetThePicture("back????.pcx", "back9999.pcx");
          break;           /*  NEWPIC   */
      case 4:
          SaveThePicture();
          break;           /*  SAVEPIC  */
      case 5:
          DoneStick();     /*  FINISH   */
          break;
    }
  }
}


/* ---------------------------------------------------------------------- */
/* The Main Menu                                                          */
/* ---------------------------------------------------------------------- */
int HotMain(int choice)
{

    char c=INVALID_CHAR;
    int oldchoice = choice,
        cx=320/6,
        cy=200/4;

    /* keys F1 through F6 toggle choices      */
    /* escape key starts or returns to picture*/
    /* enter selects the option               */

   do {
      Zap(BLKBLK);
      if (iFirstMenu == FALSE) {
        /* first time in here, load the menu into the buffer */
        /* no need to after that though... */
        strcpy(szPcxFileName,"STIXMENU.PCX");
        PcxLoad(SCREEN_ADDRESS);
        SaveScreen(lpMenuBuffer);
        iFirstMenu = TRUE;
      }
      else
        RestoreScreen(lpMenuBuffer);

      iDrawColor=WHITE_CGA;
      LineBox(PicPosition[choice][0],PicPosition[choice][1],
              PicPosition[choice][2],PicPosition[choice][3], WHITE_CGA);
      LineBox(PicPosition[choice][0]+1,PicPosition[choice][1]+1,
                PicPosition[choice][2]-1,PicPosition[choice][3]-1,
                WHITE_CGA);

      iMousex =  PicPosition[choice][0] + cx;
      iMousey =  PicPosition[choice][1] + cy;

      ShowMickey();
      if (choice!=2)
        KeyClick();
      while(c!=ENTERKEY){

         c = INVALID_CHAR;
         if (kbhit()) c=getch();
         c = HotClick(c, &choice, &oldchoice);

         if (c==ESCKEY) {
            HideMickey();
            RestoreScreen(lpScreenBuffer);

            for (;;) {
              if (kbhit()) {
                 c = getch();
                 if (c == FUNCKEY)
                    getch();
                 if (c == ESCKEY)
                   break;
              }
              if (ClickOnly())
                 break;
            }
            RestoreScreen(lpMenuBuffer);
            LineBox(PicPosition[choice][0],PicPosition[choice][1],
                PicPosition[choice][2],PicPosition[choice][3],
                WHITE_CGA);
            LineBox(PicPosition[choice][0]+1,PicPosition[choice][1]+1,
                PicPosition[choice][2]-1,PicPosition[choice][3]-1,
                WHITE_CGA);

            c = INVALID_CHAR;
            ShowMickey();
            continue;
          }

          if (c == FUNCKEY) {
              c = getch();
              switch (c) {
                case  UPARROW:
                  choice -= 3;
                  if (choice < 0) choice+= 6;
                  break;
                case  DOWNARROW:
                  choice += 3;
                  if (choice > 5) choice-= 6;
                  break;
                case  LTARROW:
                  choice -= 1;
                  if (choice < 0) choice= 5;
                  break;
                case  RTARROW:
                  choice += 1;
                  if (choice > 5) choice= 0;
                  break;
                case F1:
                case F2:
                case F3:
                case F4:
                case F5:
                case F6:
                  choice=c-59;
                  break;
                default: break;
              }
              switch (c) {
                case  UPARROW:
                case  DOWNARROW:
                case  LTARROW:
                case  RTARROW:
                case F1:
                case F2:
                case F3:
                case F4:
                case F5:
                case F6:
                  iMousex=PicPosition[choice][0]+cx;
                  iMousey=PicPosition[choice][1]+cy;
                  break;
              }
          }
          if (choice != oldchoice){
            HideMickey();
            RestoreScreen(lpMenuBuffer);
            LineBox(PicPosition[choice][0],PicPosition[choice][1],
                    PicPosition[choice][2],PicPosition[choice][3],
                    WHITE_CGA);
            LineBox(PicPosition[choice][0]+1,PicPosition[choice][1]+1,
                    PicPosition[choice][2]-1,PicPosition[choice][3]-1,
                    WHITE_CGA);
            ShowMickey();
            oldchoice=choice;
          }
        }
    } while (c != ENTERKEY);
    HideMickey();
    return (choice);
}


/* ---------------------------------------------------------------------- */
/* HotClips                                                               */
/* Sticker Selection Menu                                                 */
/* ---------------------------------------------------------------------- */
int HotClips()
{
    FILE *f1;
    char scratchbuf[128];
    int i,
        oldchoice,
        counter,
        savecounter,
        cx=320/6,
        cy=192/4;

    char c=59,
         *errmsg;

    errmsg = (char *)NULL;

    /* keys F1 through F6 toggles clip choice */
    /* enter to return to drawing             */

    Zap(BLKBLK);

    if (0 == GetWildFiles("stix????.pcx", "stix9999.pcx")) {
      errmsg = (char *)"No Stickers Found!";
      LineBox(0,0,XMAX,YMAX, CYAN_CGA);
      PCMidFont(errmsg,161,129,2, WHITE_CGA);
      PCMidFont(errmsg,160,130,2, RED_CGA);
      Bronx();
      if (FUNCKEY == getch())
        getch();
      return (INVALID);
    }

    counter = 0;
    strcpy(szPcxFileName, WildFiles[counter].name);
    PcxLoad(lpWorkBuffer);
    RestoreScreen(lpWorkBuffer);

    iCurrentChoice=0;
    oldchoice=0;
    LineBox(StickBox[iCurrentChoice][0],StickBox[iCurrentChoice][1],
            StickBox[iCurrentChoice][2],StickBox[iCurrentChoice][3], WHITE_CGA);
    TitleStuff();

    iMousex =  StickBox[iCurrentChoice][0] + cx;
    iMousey =  StickBox[iCurrentChoice][1] + cy;

    ShowMickey();

    while (c != ENTERKEY && c!= ESCKEY){

        c = INVALID_CHAR;
        if (kbhit()) c=getch();
        c = HotStick(c, &iCurrentChoice, &oldchoice);

        if (c==FUNCKEY || c==PGUP || c==PGDOWN ||
            c==BACKSPACE || c==SPACEBAR) {
          if (c==FUNCKEY) c=getch();
          switch (c) {
            case BACKSPACE:
            case SPACEBAR:
              HideMickey();
              MirrorScreen(lpWorkBuffer);
              RestoreScreen(lpWorkBuffer);
              LineBox(StickBox[iCurrentChoice][0],StickBox[iCurrentChoice][1],
                      StickBox[iCurrentChoice][2],StickBox[iCurrentChoice][3], WHITE_CGA);
              TitleStuff();
              ShowMickey();
              break;

            /* page up and down in the list */
            case HOMEKEY:
            case PGUP:
            case PGDOWN:
            case ENDKEY:
              switch (c) {
              case HOMEKEY:
                counter = 0;
                break;
              case PGUP:
                if (counter < 1)
                  counter = iWildCounter - 1;
                else
                  counter--;
                break;
              case PGDOWN:
                counter++;
                if (counter >= iWildCounter)
                  counter = 0;
                break;
              case ENDKEY:
                counter = iWildCounter - 1;
                break;
              }
              HideMickey();
              strcpy(szPcxFileName, WildFiles[counter].name);
              PcxLoad(lpWorkBuffer);
              RestoreScreen(lpWorkBuffer);

              LineBox(StickBox[iCurrentChoice][0],StickBox[iCurrentChoice][1],
                      StickBox[iCurrentChoice][2],StickBox[iCurrentChoice][3], WHITE_CGA);
              TitleStuff();
              ShowMickey();
              break;
            case  UPARROW:
              iCurrentChoice -= 3;
              if (iCurrentChoice < 0) iCurrentChoice+= 6;
              break;
            case  DOWNARROW:
              iCurrentChoice += 3;
              if (iCurrentChoice > 5) iCurrentChoice-= 6;
              break;
            case  LTARROW:
              iCurrentChoice -= 1;
              if (iCurrentChoice < 0) iCurrentChoice= 5;
              break;
            case  RTARROW:
              iCurrentChoice += 1;
              if (iCurrentChoice > 5) iCurrentChoice= 0;
              break;

            case F1:
            case F2:
            case F3:
            case F4:
            case F5:
            case F6:
              iCurrentChoice=c-59;
            default: break;
          }
          switch(c) {
          case  UPARROW:
          case  DOWNARROW:
          case  LTARROW:
          case  RTARROW:
          case F1:
          case F2:
          case F3:
          case F4:
          case F5:
          case F6:
            iMousex=StickBox[iCurrentChoice][0]+cx;
            iMousey=StickBox[iCurrentChoice][1]+cy;
          }
        }
        if (iCurrentChoice!=oldchoice){
          HideMickey();
          LineBox(StickBox[oldchoice][0],StickBox[oldchoice][1],
                  StickBox[oldchoice][2],StickBox[oldchoice][3], BLACK_CGA);
          LineBox(StickBox[iCurrentChoice][0],StickBox[iCurrentChoice][1],
                  StickBox[iCurrentChoice][2],StickBox[iCurrentChoice][3], WHITE_CGA);
          ShowMickey();
          oldchoice=iCurrentChoice;
        }
    }

    HideMickey();
    if (c != ESCKEY) {
      RestoreScreen(lpWorkBuffer);
      GetClip(StickBox[iCurrentChoice][0],StickBox[iCurrentChoice][1],
              StickBox[iCurrentChoice][2],StickBox[iCurrentChoice][3]);
    }
    return (VALID);
}


/* ---------------------------------------------------------------------- */
/* The sticker paste Menu Choice                                          */
/* ---------------------------------------------------------------------- */
void HotPaste()
{
    int x1=StickBox[iCurrentChoice][0],
        y1= StickBox[iCurrentChoice][1],
        x2=StickBox[iCurrentChoice][2],
        y2= StickBox[iCurrentChoice][3];

    int tos=0,
        bos=YMAX,
        ltedge=0,
        rtedge=XMAX;

    int width  = ((x2 - x1) + 1),
        height = (y2 - y1) + 1,
        cx =  48,
        cy =  48;

    int x=x1+(x2-x1),
        y=y1+(y2-y1),
        xtest,
        ytest;

    char c;

    int bMove = FALSE;

    Zap(BLKBLK);   /* the main screen is restored */
    ClipMove(x1,y1,x2,y2);
    iMousex = x1 + cx;
    iMousey = y1 + cy;
    ShowMickey();
    VerticalMickeyMinMax(cy, bos);

    do {

        c = INVALID_CHAR;
        if (kbhit()) c = getch();
        xtest = x1 + cx;
        ytest = y1 + cy;
        c = StickClick(c, &xtest, &ytest);

        if (c == ENTERKEY) {
          /* save the screen but don't quit */
          HideMickey();
          SaveScreen(lpScreenBuffer);
          ShowMickey();
        }

        if (c == BACKSPACE || c == SPACEBAR) {
          HideMickey();
          MirrorScreen(lpScreenBuffer);
          RestoreScreen(lpScreenBuffer);
          ShowMickey();
          bMove = TRUE;
        }

        if (xtest != (x1 + cx) || ytest!= (y1 + cy)) {
          bMove = TRUE;
          x1 = xtest - cx;
          y1 = ytest - cy;
          x2 = x1 + width - 1;
          y2 = y1 + height - 1;
          x=x1+(x2-x1);
          y=y1+(y2-y1);

        }


        if (c==FUNCKEY) {
          c=getch();
          switch(c){
              case UPARROW:   if(y1<tos+2)break;
                              y-=2;y1-=2;y2-=2;bMove = TRUE;
                              break;
              case DOWNARROW: if(y1>bos)break;
                              y+=2;y1+=2;y2+=2;bMove = TRUE;
                              break;
              case LTARROW:   if(x2<ltedge)break;
                              x-=4;x1-=4;x2-=4;bMove = TRUE;
                              break;
              case RTARROW:   if(x1>rtedge)break;
                              x+=4;x1+=4;x2+=4;bMove = TRUE;
              default:        break;
              }
        }
        if (bMove == TRUE){
          HideMickey();
          iMousex = x1 + cx;
          iMousey = y1 + cy;
          ClipMove(x1,y1,x2,y2);
          ShowMickey();
          bMove = FALSE;
        }

    } while (c!=ESCKEY);
    VerticalMickeyMinMax(tos, bos);
    HideMickey();
    return;
}


/* ---------------------------------------------------------------------- */
/* The ABC Menu Choice                                                    */
/* ---------------------------------------------------------------------- */
int HotLetters()
{
     int x=XMOS,y=YMOS;
     int scale=2,
         drop=1;
     int tos=0,bos=YMAX-(8*scale),ltedge=0,rtedge=XMAX;
     char c;
     int bMove=FALSE,i;
     int bShadow=FALSE;
     char letterbuffer[42];
     int lettercounter=0,
         xtest,
         ytest;

     Zap(BLKBLK);
     iDrawColor   = 3;
     iShadowColor = 0;

     for(i=0;i!=42;i++)letterbuffer[i]=0;
     letterbuffer[0]='+';
     RestoreScreen(lpScreenBuffer);
     PCRomFont(letterbuffer, x, y, scale, iDrawColor);
     iMousex = x;
     iMousey = y;
     ShowMickey();

     do {

         c=INVALID_CHAR;
         if (kbhit())c=getch();
         xtest = x;
         ytest = y;
         c = StickClick(c, &xtest, &ytest);
         if (xtest != x || ytest!= y) {
            x = xtest;
            y = ytest;
            bMove = TRUE;
         }

         if(c==ENTERKEY) {
           letterbuffer[lettercounter]=32;
           HideMickey();
           RestoreScreen(lpScreenBuffer);
           if(bShadow==TRUE)
             PCRomFont(letterbuffer,x+drop,y-drop,scale,
                       iShadowColor);
           PCRomFont(letterbuffer,x,y,scale, iDrawColor);
           SaveScreen(lpScreenBuffer);
           letterbuffer[lettercounter]='+';
           if(bShadow==TRUE)
             PCRomFont(letterbuffer,x+drop,y-drop,scale,
                       iShadowColor);
           PCRomFont(letterbuffer,x,y,scale, iDrawColor);
           ShowMickey();
         }

        /* make sure we are in a standard alphanumeric range */
        if (c>31 && c<123) {
          if (lettercounter<38/scale){
            letterbuffer[lettercounter]=c;
            lettercounter++;
            letterbuffer[lettercounter]='+';
            HideMickey();
            RestoreScreen(lpScreenBuffer);
            if (bShadow==TRUE)
              PCRomFont(letterbuffer,x+drop,y-drop,scale,
                        iShadowColor);
            PCRomFont(letterbuffer,x,y,scale, iDrawColor);
            ShowMickey();
          }
        }
        if (c == BACKSPACE) {
          letterbuffer[lettercounter]=32;
          lettercounter--;
          if(lettercounter<0)
            lettercounter=0;
          letterbuffer[lettercounter]='+';
          HideMickey();
          RestoreScreen(lpScreenBuffer);
          if (bShadow==TRUE)
            PCRomFont (letterbuffer,x+drop,y-drop,scale,
                       iShadowColor);
          PCRomFont(letterbuffer,x,y,scale, iDrawColor);
          ShowMickey();
        }
        if (c==FUNCKEY) {
          c=getch();
          switch (c) {
            case PGUP:
              if (scale > 23) {
                Bronx();
                break;
              }
              scale++;
              if (scale>24)scale=24;
              bos=YMAX-(8*scale);
              bMove = TRUE;
              drop=scale/2;
              if (drop>4)
                drop=4;
              break;
            case PGDOWN:
              if (scale < 2) {
                Bronx();
                break;
              }
              scale--;
              if(scale<1)scale=1;
              bos=YMAX-(8*scale);
              bMove = TRUE;
              drop=scale/2;
              if (drop<1)
                drop=1;
              break;
            case F1:
              iDrawColor=CYAN_CGA;
              bMove = TRUE;
              break;
            case F2:
              iDrawColor=RED_CGA;
              bMove = TRUE;
              break;
            case F3:
              iDrawColor=WHITE_CGA;
              bMove = TRUE;
              break;
            case F4:
              iDrawColor=BLACK_CGA;
              bMove = TRUE;
              break;
            case F5:
              if (bShadow==TRUE && iShadowColor == CYAN_CGA)bShadow = FALSE;
              else bShadow = TRUE;
              iShadowColor=CYAN_CGA;
              bMove = TRUE;
              break;
            case F6:
              if (bShadow==TRUE && iShadowColor == RED_CGA)bShadow = FALSE;
              else bShadow = TRUE;
              iShadowColor=RED_CGA;
              bMove = TRUE;
              break;
            case F7:
              if (bShadow==TRUE && iShadowColor == WHITE_CGA)bShadow = FALSE;
              else bShadow = TRUE;
              iShadowColor=WHITE_CGA;
              bMove = TRUE;
              break;
            case F8:
              if (bShadow==TRUE && iShadowColor == BLACK_CGA)bShadow = FALSE;
              else bShadow = TRUE;
              iShadowColor=BLACK_CGA;
              bMove = TRUE;
              break;
            case UPARROW:
              if(y<tos+2)break;
              y-=2;
              bMove = TRUE;
              break;
            case DOWNARROW:
              if(y>bos)break;
              y+=2;
              bMove = TRUE;
              break;
            case LTARROW:
              if(x<ltedge+4)break;
              x-=4;
              bMove = TRUE;
              break;
            case RTARROW:
              if(x>rtedge-4)break;
              x+=4;
              bMove = TRUE;
            default:
              break;
          }
        }
        if (bMove==TRUE){
          HideMickey();
          iMousex = x;
          iMousey = y;
          RestoreScreen(lpScreenBuffer);
          if (bShadow==TRUE)
            PCRomFont(letterbuffer,x+drop,y-drop,scale,
                      iShadowColor);
          PCRomFont(letterbuffer,x,y,scale,iDrawColor);
          ShowMickey();
          bMove=FALSE;
        }

     } while (c!=ESCKEY);
     HideMickey();
     return 0;
}


/* ---------------------------------------------------------------------- */
/* GetThePicture                                                          */
/* Called to build a file list and to load and provide selection for      */
/* save PCX's and also for the backgrounds shipped with this program.     */
/* ---------------------------------------------------------------------- */
int GetThePicture(char *pszSearchSpec, char *pszSearchFilter)
{
    int idx = VALID,
        oldidx = INVALID,
        dummy = 0,
        status = VALID,
        bMirror = FALSE;
    char c;

    iMenuTag = GETTHEPICTURE_MENU;

    if (GetWildFiles(pszSearchSpec, pszSearchFilter) != 0) {
      ShowMickey();
      for (;;) {

        if (idx != oldidx) {
          HideMickey();
          strcpy(szPcxFileName, WildFiles[idx].name);
          Zap(BLKBLK);
          PcxLoad(SCREEN_ADDRESS);
          TitleStuff();
          oldidx = idx;
          ShowMickey();
          bMirror = FALSE;
        }

        c = INVALID_CHAR;
        if (kbhit()) c=getch();
        c = HotStick(c, &dummy, &dummy);
        if (c == INVALID_CHAR)continue;

        if(c == ESCKEY)break;

        if (c != FUNCKEY && c!= PGDOWN && c!= PGUP) {
          if (c==ENTERKEY) {
            HideMickey();
            /* save the screen without the menu */
            PcxLoad(lpScreenBuffer);
            if (bMirror == TRUE)
              MirrorScreen(lpScreenBuffer);
            ShowMickey();
            break;
          }
          if (c == BACKSPACE || c == SPACEBAR) {
            HideMickey();
            PcxLoad(SCREEN_ADDRESS);
            if (bMirror == TRUE)
              bMirror = FALSE;
            else {
              MirrorScreen(SCREEN_ADDRESS);
              bMirror = TRUE;
            }
            TitleStuff();
            ShowMickey();
          }

        }
        else {
          if (c == FUNCKEY)
            c = getch();
          switch (c) {
            case HOMEKEY:
              idx = 0;
              break;
            case ENDKEY:
              idx = iWildCounter - 1;
              break;
            case PGUP:
              if (idx < 1)
                idx = iWildCounter - 1;
              else
                idx--;
              break;
            case PGDOWN:
              idx++;
              if (idx >= iWildCounter)
                idx = 0;
              break;
          }
        }
      }
      HideMickey();
      FreeWildFiles();
    }
    else
      status = INVALID;
    return (status);
}


/* ---------------------------------------------------------------------- */
/* SaveThePicture                                                         */
/* Automatically names and saves the current screen to PCX.               */
/* ---------------------------------------------------------------------- */
int SaveThePicture()
{
    char pszSaved[PATHLEN];
    int idx;

    /* allow up to 9999 saved images at once */
    strcpy(pszSaved,"SAVE0000.PCX");
    for (idx = 0; idx < 9999; idx++) {
      sprintf(pszSaved, "save%00004d.pcx", idx);
      if (FileExists(pszSaved))
        continue;
      MakePcxHeader();
      SaveToPcx(pszSaved);
      break;
    }
}


/* ---------------------------------------------------------------------- */
/* System Level Helper Functions                                          */
/* ---------------------------------------------------------------------- */

/* ---------------------------------------------------------------------- */
/* Initialization                                                         */
/* Called At Start of Program                                             */
/* ---------------------------------------------------------------------- */
int InitStick()
{
    int iBackGround=BLUE+INTENSE,
        iPalette=GRY,
        status;

    SetCrtMode(TEXT);
    if ((status=CheckForCGA())==FALSE) {
      printf("CGA required\n");
      exit(1);
    }
    if ((status = SetBuffers()) == FALSE) {
      printf("Memory Allocation Error\n");
      exit(2);
    }
    SetCrtMode(iScreenMode);
    if(iScreenMode==CGA_320) {
      SetScreenColor(iBackGround,iPalette);
    }
    MakeMickey();
    HiMickey();

    strcpy(szPcxFileName, "STIXTIT.PCX");
    if (PcxLoad(lpWorkBuffer) == VALID)
      RestoreScreen(lpWorkBuffer);
    else {
      PCMidFont("STIX",162,18,9, WHITE_CGA);
      PCMidFont("STIX",160,20,9, RED_CGA);

      PCMidFont("Sticker Draw",161,89,3, RED_CGA);
      PCMidFont("Sticker Draw",160,90,3, CYAN_CGA);

      PCMidFont("Version 2.0",160,119,2, WHITE_CGA);
      PCMidFont("Version 2.0",160,120,2, RED_CGA);
      PCMidFont("(C)Copyright 1990-1999",161,139,1, RED_CGA);
      PCMidFont("(C)Copyright 1990-1999",160,140,1, CYAN_CGA);
      PCMidFont("by Bill Buckels",161,151,2, WHITE_CGA);
      PCMidFont("by Bill Buckels",160,152,2, RED_CGA);
    }
    LineBox(0,0,XMAX,YMAX, CYAN_CGA);
    for (;;) {
      if (kbhit()) {
        if (getch() == FUNCKEY)
          getch();
        break;
      }
      if (ClickOnly())
        break;
    }
    Zap(BLKBLK);
    SaveScreen(lpScreenBuffer);
    return 0;
}

/* ---------------------------------------------------------------------- */
/* DoneStick()                                                            */
/* Called At End of Program                                               */
/* ---------------------------------------------------------------------- */
void DoneStick()
{

    SetCrtMode(TEXT);
    if (FileExists("RUNFIRST.EXE"))
      system("RUNFIRST.EXE");
    else
      puts("Have a Nice Dos!");
    exit(0);
}


/* ---------------------------------------------------------------------- */
/* TitleStuff                                                             */
/* Sets up titlestuff and menustuff after loading a library image.        */
/* Called when loading Clips, Backgrounds, and Saved Images.              */
/* ---------------------------------------------------------------------- */
void TitleStuff(void)
{
    int i;
    if (iMenuTag == HOTCLIPS_MENU) {
      for (i=0; i<6; i++)
      {
        FilledBox (TitleBox[i][0],TitleBox[i][1],
              TitleBox[i][2],TitleBox[i][3], RED_CGA);

        LineBox(TitleBox[i][0],TitleBox[i][1],
                TitleBox[i][2],TitleBox[i][3], CYAN_CGA);

        PCMidFont(ClipText[i],
                  TitleBox[i][0]+((TitleBox[i][2]-TitleBox[i][0])/2),
                  TitleBox[i][1]+((TitleBox[i][3]-TitleBox[i][1])/2)-3,
                  1, CYAN_CGA);
      }
    }

    FilledBox(TitleBox[6][0],TitleBox[6][1],
         TitleBox[6][2],TitleBox[6][3], CYAN_CGA);
    FilledBox(TitleBox[7][0],TitleBox[7][1],
         TitleBox[7][2],TitleBox[7][3], RED_CGA);

   PCMidFont(ClipText[6],
             TitleBox[6][0]+((TitleBox[6][2]-TitleBox[6][0])/2),
             TitleBox[6][1]+((TitleBox[6][3]-TitleBox[6][1])/2)-3,
             1, WHITE_CGA);
   PCMidFont(ClipText[7],
             TitleBox[7][0]+((TitleBox[7][2]-TitleBox[7][0])/2),
             TitleBox[7][1]+((TitleBox[7][3]-TitleBox[7][1])/2)-3,
             1, WHITE_CGA);

}


/* ---------------------------------------------------------------------- */
/* SetBuffers                                                             */
/* Allocates Memory used by this program.                                 */
/* ---------------------------------------------------------------------- */
int SetBuffers()
{
   if ((lpScreenBuffer = calloc(SCREENBUF_SIZE, 1)) == NULL)
     return FALSE;

   if ((lpMenuBuffer = calloc(SCREENBUF_SIZE, 1)) == NULL)
     return FALSE;

   if ((lpWorkBuffer = calloc(SCREENBUF_SIZE, 1)) == NULL)
     return FALSE;

   if ((lpClipBuffer = calloc(CLIPBUF_SIZE, 1)) == NULL)
     return FALSE;

   return TRUE;
}


/* ---------------------------------------------------------------------- */
/* File Routines                                                          */
/* ---------------------------------------------------------------------- */

/* ---------------------------------------------------------------------- */
/* FreeWildFiles                                                          */
/* Reset the file list and free memory                                    */
/* ---------------------------------------------------------------------- */
void FreeWildFiles()
{
  if (WildFiles != (WILDFILES *)NULL)
    free(WildFiles);

  WildFiles = (WILDFILES *)NULL;
  iWildCounter = 0;
}

/* ---------------------------------------------------------------------- */
/* IsFormatValid                                                          */
/* Test the filemask to make sure that its name matches the pattern       */
/* ---------------------------------------------------------------------- */
int IsFormatValid(char *pszFileName, char *pszFormatSpec)
{
    int len,
        len2,
        idx,
        iRetVal;

    char ch,
         fmt;

    iRetVal = TRUE;

    if (NULL != pszFormatSpec) {
      len  = strlen(pszFormatSpec);
      len2 = strlen(pszFileName);
      if (len != len2)
        iRetVal = FALSE;
      else {
        for (idx = 0; idx < len; idx++) {
          ch = tolower(pszFileName[idx]);
          fmt = pszFormatSpec[idx];
          switch (fmt) {
            case '9':
              if (ch < '0' || ch > '9')
                iRetVal = FALSE;
                break;
            case 'X':
              if (ch < 'A' || ch > 'Z')
                iRetVal = FALSE;
                break;
            case '?':   /* allow individual wildcards */
              break;
            default:
              if (fmt != ch)
                iRetVal = FALSE;
          }
          if (iRetVal == FALSE)
            break;
        }
      }
    }
    return (iRetVal);
}


/* ---------------------------------------------------------------------- */
/* GetWildFiles                                                           */
/* Build the file list, allocate memory                                   */
/* ---------------------------------------------------------------------- */
int GetWildFiles(char *pszSearchSpec, char *pszFormatSpec)
{

  FreeWildFiles();
  iWildCounter = 0;
  if(_dos_findfirst(pszSearchSpec,_A_NORMAL,&WildCard) == 0)
  {
    if (TRUE == IsFormatValid(WildCard.name, pszFormatSpec))
      iWildCounter++;
    while(_dos_findnext(&WildCard) == 0)
      if (TRUE == IsFormatValid(WildCard.name, pszFormatSpec))
        iWildCounter++;
    if (iWildCounter) {
      if((WildFiles=malloc(sizeof(WILDFILES)*iWildCounter))==NULL) {
        iWildCounter = 0;
      }
      else {
        iWildCounter = 0;
        _dos_findfirst(pszSearchSpec,_A_NORMAL,&WildCard);
        for (;;) {
          if (TRUE == IsFormatValid(WildCard.name, pszFormatSpec)) {
            strcpy(WildFiles[iWildCounter].name, WildCard.name);
            iWildCounter++;
          }
          if(_dos_findnext(&WildCard) != 0)
            break;
        }
      }
      if( iWildCounter >1)
        qsort(WildFiles,iWildCounter,sizeof(WILDFILES),compare);
    }
  }
  return (iWildCounter);
}


/* ---------------------------------------------------------------------- */
/* type conversion functions used for pcx validation                      */
/* ---------------------------------------------------------------------- */
unsigned int byteword(unsigned char a, unsigned char b){return b<<8|a;}
unsigned char lsb(unsigned int word){ return word &0xff;}
unsigned char msb(unsigned int word){ return word >>8;}

/* ---------------------------------------------------------------------- */
/* CheckForPcx                                                            */
/* Header Validation function called during loading of a pcx file.        */
/* ---------------------------------------------------------------------- */
int CheckForPcx(char *name)
{
    FILE *fp; /* reads a ZSOFT .PCX header but ignores the color map */
    int i;    /* we only want CGA compatible full screens.           */

    unsigned int zsoft,
                 version,
                 codetype,
                 pixbits;
    unsigned int xmin, ymin, xmax, ymax,
                 hres, vres,
                 no_planes, bytesperline;
    int status = VALID;

    /* read the file header */
    if((fp=fopen(name,"rb"))==NULL)return INVALID;
    for (i=0 ; i<128; i++)szPcxHeader[i]=fgetc(fp);
    fclose(fp);

    zsoft   =szPcxHeader[0];
    version =szPcxHeader[1];   /* don't care about version */
    codetype=szPcxHeader[2];
    pixbits =szPcxHeader[3];

    if (zsoft!=10 || codetype!=1 || (pixbits != 2 && pixbits != 1))
       status = INVALID;
    else {

      xmin=byteword(szPcxHeader[4],szPcxHeader[5]);
      ymin=byteword(szPcxHeader[6],szPcxHeader[7]);
      xmax=byteword(szPcxHeader[8],szPcxHeader[9]);
      ymax=byteword(szPcxHeader[10],szPcxHeader[11]);

      /* don't care about resolution of source device */
      hres=byteword(szPcxHeader[12],szPcxHeader[13]);
      vres=byteword(szPcxHeader[14],szPcxHeader[15]);

      no_planes=szPcxHeader[65];
      bytesperline=byteword(szPcxHeader[66],szPcxHeader[67]);

      if (xmin != 0 || ymin != 0  ||
          (xmax != XMAX && xmax!=639) || ymax != YMAX ||
          no_planes!=1 || bytesperline !=80)
        status = INVALID;

      /* we can ignore the color map since we        */
      /* are limiting ourselves to CGA modes         */
      /* so we will not handle over 2-bits per pixel */
    }

    return status;
}


/* ---------------------------------------------------------------------- */
/* PcxLoad                                                                */
/* Load a CGA pcx into the buffer specified (usually direct to screen)    */
/* ---------------------------------------------------------------------- */
int PcxLoad(unsigned char *crt)
{
    unsigned int byteoff=0,inleaf=8192,packet,width=0;
    FILE *fp;
    unsigned char byte,bytecount;
    long wordcount,target;

    if(CheckForPcx(szPcxFileName) == INVALID)return INVALID;
    if((fp = fopen(szPcxFileName,"rb"))==NULL) return INVALID;

    target = filelength(fileno(fp));
    for(wordcount=0;wordcount!=128;wordcount++)byte=fgetc(fp);

    do{ bytecount=1;                          /* start with a seed count */
        byte=fgetc(fp);
        wordcount++;
                                              /* check to see if its raw */
        if (0xC0 == (0xC0 &byte)){            /* if its not, run encoded */
          bytecount= 0x3f &byte;
          byte=fgetc(fp);
          wordcount++;
        }
        for (packet=0;packet<bytecount;packet++) {
          if (width<80) {
            crt[byteoff]=byte;
            width++;
            byteoff++;
          }
          else {
            crt[inleaf]=byte;
            inleaf++;
            width++;
            if (width>XMOS)
              width=0;
          }
        }
    } while (wordcount<target);
    fclose(fp);
    return(VALID);
}


/* ---------------------------------------------------------------------- */
/* MakePcxHeader                                                          */
/* makes a standard header for a .PCX CGA FULL-SCREEN DUMP FILE           */
/* ---------------------------------------------------------------------- */
int MakePcxHeader()
{
    int i;
    unsigned char zsoft=10,version=3,codetype=1,pixbits=2;
    unsigned int  xmin=0, ymin=0, xmax=XMAX, ymax=YMAX;
    unsigned int  hres=320, vres=200;
    unsigned char no_planes=1;
    unsigned int  bytesperline=80;

    for (i=0; i<128; i++)
     szPcxHeader[i]=0;/* pad the header with nulls */

    if (iScreenMode == HI_RES) {
      pixbits=pixbits/2;
      xmax=((xmax+1)*2)-1;
      hres=hres*2;
    }

    szPcxHeader[0]=zsoft;
    szPcxHeader[1]=version;
    szPcxHeader[2]=codetype;
    if(iScreenMode==HI_RES)pixbits=1;
    szPcxHeader[3]=pixbits;
    szPcxHeader[4] =lsb(xmin);
    szPcxHeader[5] =msb(xmin);
    szPcxHeader[6] =lsb(ymin);
    szPcxHeader[7] =msb(ymin);
    szPcxHeader[8] =lsb(xmax);
    szPcxHeader[9] =msb(xmax);
    szPcxHeader[10]=lsb(ymax);
    szPcxHeader[11]=msb(ymax);
    szPcxHeader[12]=lsb(hres);
    szPcxHeader[13]=msb(hres);
    szPcxHeader[14]=lsb(vres);
    szPcxHeader[15]=msb(vres);
    szPcxHeader[65]=no_planes;
    szPcxHeader[66]=lsb(bytesperline);
    szPcxHeader[67]=msb(bytesperline);
    return VALID;

}


/* ---------------------------------------------------------------------- */
/* SaveToPcx                                                              */
/* Save a 4-Color PCX File.                                               */
/* ---------------------------------------------------------------------- */
int SaveToPcx(char *pszOutFile)
{
    unsigned char crtbuf[80];
    unsigned char leafbuf[80];
    unsigned int  offset=0;
    unsigned int  inleaf=8192;
    int packet = 80, i;

    if ((pcxfile=fopen(pszOutFile,"wb")) != NULL) {
       /* write the header */
       for(i=0;i!=128;i++)fputc(szPcxHeader[i],pcxfile);
       for(offset=0;offset<8000;offset+=packet){
         memcpy(crtbuf,lpScreenBuffer+offset,packet);
         memcpy(leafbuf,lpScreenBuffer+inleaf,packet);
         encline(crtbuf,packet);
         encline(leafbuf,packet);
         inleaf+=80;
       }
       fclose(pcxfile);
    }
    return (0);
}


/* ---------------------------------------------------------------------- */
/* encodes a raw line and writes it out to disk.                          */
/* required to write the PCX files saved by this program.                 */
/* ---------------------------------------------------------------------- */
int encline(unsigned char *inbuff, int inlen)
{
    unsigned char current,last;
    int srcindex,i;
    register int total;
    register unsigned char runcount;

    total=0;
    last = *(inbuff);
    runcount=1;

    for(srcindex=1;srcindex!=inlen;srcindex++){
      current= *(++inbuff);
      if(current==last){
        runcount++;
        if(runcount==63){
            if(!(i=encput(last,runcount)))
            return(0);
            total+=i;
            runcount=0;
        }
      }
      else {
        if(runcount){
            if(!(i=encput(last,runcount)))
            return(0);
            total+=i;
        }
        last=current;
        runcount=1;
      }
   }

  if (runcount) {
    if (!(i=encput(last,runcount)))
      return(0);
    return(total+i);
  }
  return (total);

}


/* ---------------------------------------------------------------------- */
/* the writer for the encline function                                    */
/* ---------------------------------------------------------------------- */
int encput(unsigned char byt, unsigned char cnt)
{
  if (cnt){
    if((cnt==1)&& (0xc0 != (0xc0 &byt))){
      if (EOF == fputc((int)byt,pcxfile))
         return(0);
      return(1);
    }
    else {
      if (EOF==fputc((int)0xc0|cnt,pcxfile))
        return(0);

      if(EOF==fputc((int)byt,pcxfile))
        return(0);

      return(2);
    }
  }
  return(0);
}


/* required for the quicksort */
int compare(arg1,arg2)
char **arg1, **arg2;
{
   return(memicmp(arg1,arg2,sizeof(WILDFILES)));
}

/* ---------------------------------------------------------------------- */
/* Graphics Core Routines                                                 */
/* ---------------------------------------------------------------------- */

/* ---------------------------------------------------------------------- */
/* Check for CGA                                                          */
/* probably not necessary in this day and age since all current video     */
/* cards will do a minimum of VGA and therefore support the old CGA mode. */
/* ---------------------------------------------------------------------- */
int CheckForCGA()
{
    /* clear the screen and try to read a whitespace from the
       address for the CGA. if its not there return 0 and split
       because the rest of the stuff won't work. */

  unsigned char *crt =(unsigned char *) 0xB8000000l;
  union REGS reg;

  reg.h.ah = 6;
  reg.h.al = 0;
  reg.h.ch = 0;
  reg.h.cl = 0;
  reg.h.dh = 24;
  reg.h.dl = 79;
  reg.h.bh = 0x07;
  int86(0x10, &reg, &reg);

  if(crt[0]==32)return TRUE;
  return FALSE;
}


/* ---------------------------------------------------------------------- */
/* SetCrtMode                                                             */
/* Sets the video mode using the standard bios int10h call.               */
/* ---------------------------------------------------------------------- */
void SetCrtMode(int m)
{
    union REGS rin,rout;
    rin.h.ah = 0;
    rin.h.al = m;
    int86(0x10,&rin,&rout);

}


/* ---------------------------------------------------------------------- */
/* SetScreenColor                                                         */
/* Sets the CGA Color and Palette using the standard bios int10h call.    */
/* ---------------------------------------------------------------------- */
int SetScreenColor(unsigned background, unsigned palette)
{
    union REGS rin,rout;

    rin.h.ah = 11;
    rin.h.bh = 0;
    rin.h.bl = (unsigned char)background;
    int86(0x10,&rin,&rout);
    rin.h.bh = 1;
    rin.h.bl = (unsigned char)palette;
    int86(0x10,&rin,&rout);
    return 0;
}


/* ---------------------------------------------------------------------- */
/* Zap                                                                    */
/* Clears the screen to a CGA Color using the standard bios int10h call.  */
/* ---------------------------------------------------------------------- */
void Zap(char zapcolor)
{
    union REGS rin,rout;

    rin.x.dx= 0x1828;
    rin.h.bh= zapcolor;
    rin.x.ax= 0x0600;
    rin.x.cx= 0;
    int86(0x10,&rin,&rout);
}


/* ---------------------------------------------------------------------- */
/* PutDot                                                                 */
/* Writes a pixel at x,y (col, row) in a specified color.                 */
/* ---------------------------------------------------------------------- */
void PutDot(col,row,uiColor)
unsigned int col,row;
unsigned int uiColor;
{
   unsigned char *crt =
   (unsigned char *)0xB8000000l+(0x2000 * (row % 2)) + (80 * (row/2)) + col/4;
   unsigned int mask,temp;
   unsigned char color = (unsigned char)uiColor;

   if (row > YMAX)
     return;

   mask= 0xc0 >> ((col%4)*2);
   temp=*crt;

   *crt= temp & ~mask;
   mask= (color & 0x03) << (6-(col%4)*2);
   temp=*crt;
   *crt= temp|mask;
}


/* ---------------------------------------------------------------------- */
/* LineBox                                                                */
/* Create an outlined box using the color specified.                      */
/* ---------------------------------------------------------------------- */
void LineBox(int x1, int y1, int x2, int y2, int linecolor)
{
    int x,y;

    for (x=x1; x<x2+1; x++)
      PutDot(x, y1, linecolor);
    for (y=y1+1; y<y2; y++) {
      PutDot(x1, y, linecolor);
      PutDot(x2, y, linecolor);
    }
    for (x=x1; x!=x2+1; x++)
      PutDot(x, y2, linecolor);
}


/* ---------------------------------------------------------------------- */
/* FilledBox                                                              */
/* Create a filled box using the color specified.                         */
/* ---------------------------------------------------------------------- */
void FilledBox(int x1,int y1,int x2,int y2,int iColor)
{
   int x,y;

   /* swap coordinates if out of order */
   if (x2<x1) {
     x=x2;
     x2=x1;
     x1=x;
   }
   if (y2<y1) {
     y=y2;
     y2=y1;
     y1=y;
   }

   x2++;
   y2++;

   for (y=y1; y<y2; y++)
     for (x=x1; x<x2; x++)
       PutDot(x,y,iColor);
}


/* ---------------------------------------------------------------------- */
/* PCRomFont                                                              */
/* Uses the rom font as a template for a bitmap font.                     */
/* This allows us to map a font using scaling techniques on any PC        */
/* without the need for an external font file.                            */
/* ---------------------------------------------------------------------- */
void PCRomFont(unsigned char *str,
               int xorigin, int yorigin, int scale, int fontcolor)
{
    int scanline,
        yreg=yorigin,
        xreg=xorigin,
        byt,
        character,
        nibble;
    int x,y;
                                            /* flags etcetera */

    unsigned char *romfont=(unsigned char *) 0xffa6000el;
    /* a pointer to the 8 x 8 BITMAPPED font in the PC ROM    */

    int target = strlen(str);  /* string length               */

    for (scanline=0;scanline<8;scanline++)/* finish the current scanline  */
    {                                     /*  before advancing to the next*/
      for (byt=0;byt<target;byt++)     /* run the scanline             */
      {
        /* get the bitmap  */
        character = romfont[(str[byt]&0x7f)*8+scanline];
        for (nibble=0;nibble<8;nibble++)
        {
          xreg+=scale;
          /* chew the byte to bits and lite the pixel if it's a swallow  */
          if (str[byt]!=CRETURN && str[byt]!=LFEED)
            if (character & 0x80>>nibble)
              for (x=0; x!=scale; x++)
                for (y=0;y!=scale;y++)
                  PutDot(xreg+x,yreg+y,fontcolor);
        }
      }
      yreg+=scale;
      xreg=xorigin;
    }

}


/* ---------------------------------------------------------------------- */
/* PCMidFont                                                              */
/* Maps to PCRomfont, uses a centre-justified x-corrdinate.               */
/* ---------------------------------------------------------------------- */
void PCMidFont(unsigned char *str, int xmiddle, int yorigin,
               int scale, int fontcolor)
{   /* centre justified string */
    PCRomFont(str,(xmiddle-(4*(strlen(str))*scale)),yorigin,scale,
              fontcolor);
}


/* ---------------------------------------------------------------------- */
/* SaveScreen and Restore Screen                                          */
/* Complimentary functions to move from the screen to a save buffer       */
/*   and from the buffer onto the screen. Used to preserve the original   */
/*   contents prior to making some change that we may want to undo.       */
/* ---------------------------------------------------------------------- */
void SaveScreen(char *ptr)
{   unsigned char *crt =(unsigned char *) 0xB8000000l;
    memmove(ptr, crt, SCREENBUF_SIZE);
}

void RestoreScreen(char *ptr)
{   unsigned char *crt =(unsigned char *) 0xB8000000l;
    memmove(crt, ptr, SCREENBUF_SIZE);
}


/* ---------------------------------------------------------------------- */
/* MirrorScreen                                                           */
/* Mirror the screen buffer left to right and right to left about the     */
/* middle axis.                                                           */
/* ---------------------------------------------------------------------- */
void MirrorScreen(char *ptr)
{
    char *crt,
         *inleaf,
         *crtptr,
         tempx,
         tempx1;

    unsigned y, x, x1, idx, offset;
    union cgapixel Clip, Crt;

    crt    = (char *)&ptr[0];
    inleaf = (char *)&ptr[0x2000];
    offset = 0;

    for (y = 0; y < 100; y++) {
      for (idx = 0; idx < 2; idx++) {
        if (idx == 0)
          crtptr = (char *)&crt[offset];
        else
          crtptr = (char *)&inleaf[offset];
        /* move through each scanline from either end to the middle */
        /* and mirror each byte with its complimentary twin.        */
        x1 = 80;
        for (x = 0; x < 40; x++) {
          tempx  = crtptr[x];        /* left side */
          x1--;
          tempx1 = crtptr[x1];       /* right side */
          if (tempx == BLKBLK || tempx == BLUBLK ||
              tempx == REDBLK || tempx == WHTBLK) {
            crtptr[x1] = tempx;
          }
          else {
            Crt.byt  = tempx;
            Clip.pixel.x1 = Crt.pixel.x4;
            Clip.pixel.x2 = Crt.pixel.x3;
            Clip.pixel.x3 = Crt.pixel.x2;
            Clip.pixel.x4 = Crt.pixel.x1;
            crtptr[x1] = Clip.byt;
          }
          if (tempx1 == BLKBLK || tempx1 == BLUBLK ||
              tempx1 == REDBLK || tempx1 == WHTBLK) {
            crtptr[x] = tempx1;
          }
          else {
            Crt.byt  = tempx1;
            Clip.pixel.x1 = Crt.pixel.x4;
            Clip.pixel.x2 = Crt.pixel.x3;
            Clip.pixel.x3 = Crt.pixel.x2;
            Clip.pixel.x4 = Crt.pixel.x1;
            crtptr[x] = Clip.byt;
          }
        }
      }
      offset += 80;
    }
    return;
}


/* ---------------------------------------------------------------------- */
/* GetClip                                                                */
/* Get a clip from part of the screen. Called when selecting stickers.    */
/* ---------------------------------------------------------------------- */
void GetClip(int x1,int y1,int x2,int y2)
{
    unsigned char *crt =
    (unsigned char *)0xB8000000l+(0x2000 * (y1 % 2)) + (80 * (y1/2)) + x1/4;
    unsigned char *inleaf =
    (unsigned char *)0xB8000000l+(0x2000*((y1+1)% 2))+(80*((y1+1)/2))+ x1/4;

    unsigned int width=  ((x2+1)-x1)/4, /* 2 bits per pixel */
                 height= ((y2+1)-y1),
                 charcount=0,
                 offset=0;
    int y;

    for (y=0;y<height;y+=2){
      if (y1 > YMAX)
        break;
      memmove(&lpClipBuffer[charcount],&crt[offset],width);
      charcount+=width;
      memmove(&lpClipBuffer[charcount],&inleaf[offset],width);
      charcount+=width;
      offset+=80;
      y1+=2;
    }

}


/* ---------------------------------------------------------------------- */
/* PutClip                                                                */
/* Complementary function to get clip. Put a sticker on the screen.       */
/* ---------------------------------------------------------------------- */
void PutClip(int x1,int y1,int x2,int y2)
{
    unsigned char *crt =
    (unsigned char *)0xB8000000l+(0x2000 * (y1 % 2)) + (80 * (y1/2)) + x1/4;
    unsigned char *inleaf =
    (unsigned char *)0xB8000000l+(0x2000*((y1+1)% 2))+(80*((y1+1)/2))+ x1/4;
    unsigned char *cliptr,
                  *crtptr;
    unsigned int width=  ((x2+1)-x1)/4, /* 2 bits per pixel */
                 height= ((y2+1)-y1),
                 charcount=0,
                 offset=0,
                 x;
    int y,
        idx;

    union cgapixel Clip, Crt;

    for (y = 0; y < height; y+=2){
      if (y1 > YMAX)
        break;
      for (idx = 0; idx < 2; idx++) {
          /* deal with the CGA interleaf */
          if (idx == 0)
            crtptr = (char *)&crt[offset];
          else
            crtptr = (char *)&inleaf[offset];

          cliptr = (char *)&lpClipBuffer[charcount];
          charcount+= width;

          for (x = 0; x < width; x++){
            /* To avoid jaggy edges and to make this
               reasonably efficient...
               if the byte we are copying is not black,
                 if masking is unnecessary or the background is black
                 then just copy the entire byte.
                 otherwise we need to unmask each pixel.
            */

            if (*cliptr != BLKBLK) {
              if (*crtptr == BLKBLK || *cliptr == BLUBLK ||
                *cliptr == REDBLK || *cliptr == WHTBLK) {
                *crtptr = *cliptr;
              }
              else {
                Crt.byt  = *crtptr;
                Clip.byt = *cliptr;
                if (Clip.pixel.x1)
                  Crt.pixel.x1 = Clip.pixel.x1;
                if (Clip.pixel.x2)
                  Crt.pixel.x2 = Clip.pixel.x2;
                if (Clip.pixel.x3)
                  Crt.pixel.x3 = Clip.pixel.x3;
                if (Clip.pixel.x4)
                  Crt.pixel.x4 = Clip.pixel.x4;
                *crtptr = Crt.byt;
              }
            }
            if (x < width) {
              *crtptr++;
              *cliptr++;
            }
          }
      }
      offset+=80;
      y1 += 2;
    }
    return;
}


/* ---------------------------------------------------------------------- */
/* ClipMove                                                               */
/* Restore the old screen and put the clip at the new position.           */
/* ---------------------------------------------------------------------- */
int ClipMove(int x1,int y1,int x2, int y2)
{   RestoreScreen(lpScreenBuffer);
    PutClip(x1,y1,x2,y2);
}


/* ----------------------------------------------------------------------- */
/* Sound Functions                                                         */
/* ----------------------------------------------------------------------- */
#define TIMEOUT 0x2c00

int sound(freq,tlen)
int freq;         /* freq of sound in Hertz                  */
int tlen;         /* duration of sound 18.2 ticks per second */
{
    union REGS inregs,outregs;
    unsigned frdiv = 1331000L/freq;   /* timer divisor */
    int seed,hiseed,hold=0,setting;

    if(!iSoundFlag)return 0;

    if(freq!=32767)
    {
      outp(0x43,0xB6) ;            /* write timer mode register */
      outp(0x42,frdiv & 0x00FF);   /* write divisor LSB */
      outp(0x42,frdiv >> 8);       /* write divisor MSB */
      setting= inp(0x61);          /* get current port B setting */
      outp(0x61,setting | 0x03);   /* turn speaker on */
    }

    if (tlen>0) {
      tlen=((tlen*1000)/182); /* convert from 18.2 ticks to 100ths */
      inregs.x.ax= TIMEOUT;
      int86(0x21,&inregs,&outregs);
      seed=(outregs.x.dx)&0xff;
 
      while(hold<tlen)
      {

        inregs.x.ax = TIMEOUT;
        int86(0x21,&inregs,&outregs);
        if(seed!=(outregs.x.dx&0xff))
        {
          hiseed= (outregs.x.dx)&0xff;
          if(hiseed<seed)hiseed+=100;
          hold+=(hiseed-seed);
          seed =(outregs.x.dx)&0xff;
        }
      }
    }

    if(freq!=32767)
    {
      /* restore port B setting */
      outp(0x61,setting);
    }
    return 0;
}

int KeyClick(void)
{
  sound(523,1);
  sound(659,1);
  sound(784,1);
  return (0);
}

int Bronx(void)
{
  sound(60,4);
  sound(40,8);
  return (0);
}

/* ---------------------------------------------------------------------- */
/* Mouse Functions and Mouse Navigation Helper Functions                  */
/* ---------------------------------------------------------------------- */


/* ---------------------------------------------------------------------- */
/* PicMouse - Called by HotClick to return a position in the mainmenu.    */
/* (HotClick is called by HotMain)                                        */
/* ---------------------------------------------------------------------- */
int PicMouse(int x1, int y1)
{
   int i=1;

   if(x1 < PicPosition[1][0])i=0;
   if(x1 > PicPosition[1][2])i=2;

   if (y1 > YMOS)i+=3;
   return i;
}


/* ---------------------------------------------------------------------- */
/* Hotclick is called by HotMain to return mouse status and cursor        */
/* position. Mouse status is mapped to equivalent key code returns.       */
/* ---------------------------------------------------------------------- */
int HotClick(int iRetVal, int *position, int *oldposition)
{
  union REGS inregs, outregs; /* mouse registers */

  if (bMickeyMouse == TRUE) {

    inregs.x.ax = 3;
    int86(0x33,&inregs,&outregs);
    iMousex=outregs.x.cx/2;
    iMousey=outregs.x.dx;

    position[0] = PicMouse(iMousex,iMousey);

    if(outregs.x.bx)
     {
        if (outregs.x.bx == 1)
          iRetVal = ENTERKEY;
        else if (outregs.x.bx == 2)
          iRetVal = ESCKEY;

        position[0]=oldposition[0];
        while(outregs.x.bx)int86(0x33,&inregs,&outregs);
      }
   }
   return (iRetVal);
}


/* ---------------------------------------------------------------------- */
/* Clickonly - returns a left or a right mouse click as a key value       */
/*             if the mouse is installed.                                 */
/* ---------------------------------------------------------------------- */
int ClickOnly()
{
  union REGS inregs, outregs; /* mouse registers */
  int iRetVal = 0;

  if (bMickeyMouse == TRUE) {

    inregs.x.ax = 3;
    int86(0x33,&inregs,&outregs);

    if(outregs.x.bx)
     {
        if (outregs.x.bx == 1)
          iRetVal = ENTERKEY;            /* left button */
        else if (outregs.x.bx == 2)
          iRetVal = ESCKEY;              /* right button */

        while(outregs.x.bx)int86(0x33,&inregs,&outregs);
      }
   }
   return (iRetVal);
}


/* ---------------------------------------------------------------------- */
/* StickMouse - Called by Hotstick (below)                                */
/* Called from the sticker selection menu to return the mouse position.   */
/* ---------------------------------------------------------------------- */
int StickMouse(int x1, int y1)
{
   int i;
   if (y1 > 191) {
      i = 7;
      if (x1 < XMOS)
        i = 6;
   }
   else {
     i = 1;
     if(x1 < StickBox[1][0])i=0;
     if(x1 > StickBox[1][2])i=2;
     if (y1 > StickBox[1][3])
       i+=3;
   }
   return i;
}

/* ---------------------------------------------------------------------- */
/* Mouse Helper Function                                                  */
/* Called When Selecting Stickers                                         */
/* ---------------------------------------------------------------------- */
int HotStick(int iRetVal, int *position, int *oldposition)
{
  union REGS inregs, outregs; /* mouse registers */
  int choice;

  if (bMickeyMouse == TRUE) {

    inregs.x.ax = 3;
    int86(0x33,&inregs,&outregs);
    iMousex=outregs.x.cx/2;
    iMousey=outregs.x.dx;

    choice = StickMouse(iMousex,iMousey);
    if (choice < 6)
       position[0] = choice;

    if(outregs.x.bx)
     {
        if (outregs.x.bx == 1) {        /* right button */
          iRetVal = ENTERKEY;
          position[0]=oldposition[0];
          if (choice == 7)
            iRetVal = PGDOWN;
        }
        else if (outregs.x.bx == 2) {   /* left button */
          iRetVal = ESCKEY;
          if (choice == 7)
             iRetVal = PGUP;
        }

        while(outregs.x.bx)int86(0x33,&inregs,&outregs);
      }
   }
   return (iRetVal);
}

/* ---------------------------------------------------------------------- */
/* Mouse Helper Function                                                  */
/* Called When Pasting Stickers                                           */
/* ---------------------------------------------------------------------- */
int StickClick(int iRetVal, int *xtest, int *ytest)
{
  union REGS inregs, outregs; /* mouse registers */

  if (bMickeyMouse == TRUE) {
    inregs.x.ax = 3;
    int86(0x33,&inregs,&outregs);
    if(outregs.x.bx)
     {
        if (outregs.x.bx == 1)
          iRetVal = ENTERKEY;
        else if (outregs.x.bx == 2)
          iRetVal = ESCKEY;
        while(outregs.x.bx)int86(0x33,&inregs,&outregs);
      }
      else {
        iMousex=(outregs.x.cx/8)*4;  /* using multiples of 4 as xterm */
        iMousey=(outregs.x.dx/2)*2;  /* using multiples of 2 as yterm */
        xtest[0]=iMousex;
        ytest[0]=iMousey;
      }
   }
   return (iRetVal);    /* if no mouse, just pass the keypress through */
}


/* ---------------------------------------------------------------------- */
/* HideMickey                                                             */
/* Hide the mouse cursor.                                                 */
/* ---------------------------------------------------------------------- */
int HideMickey()
{
  union REGS inregs, outregs; /* mouse registers */
  if (bMickeyMouse == TRUE)
    {
      inregs.x.ax = 2;
      int86(0x33,&inregs,&outregs);
    }
  return bMickeyMouse;
}


/* ---------------------------------------------------------------------- */
/* ShowMickey                                                             */
/* Show (Unhide) The Mouse Cursor.                                        */
/* ---------------------------------------------------------------------- */
int ShowMickey()
{

  union REGS inregs, outregs; /* mouse registers */

  if (bMickeyMouse == FALSE)
     return FALSE;

   inregs.x.ax = 4;
   inregs.x.cx = iMousex*2;
   inregs.x.dx = iMousey;
   int86(0x33,&inregs,&outregs);   /* start */

   inregs.x.ax = 1;
   int86(0x33,&inregs,&outregs);   /* show  cursor*/
   return TRUE;
}


/* ---------------------------------------------------------------------- */
/* VerticalMickeyMinMax                                                   */
/* Set vertical limits of virtual mouse screen.                           */
/* Required to limit pasting of stickers in viewport.                     */
/* ---------------------------------------------------------------------- */
int VerticalMickeyMinMax(int ymin, int ymax)
{
  union REGS inregs, outregs; /* mouse registers */

  if (bMickeyMouse == FALSE)
     return FALSE;

   inregs.x.ax = 8;
   inregs.x.cx = ymin;
   inregs.x.dx = ymax;
   int86(0x33,&inregs,&outregs);   /* start */
   return TRUE;
}


/* ---------------------------------------------------------------------- */
/* MakeMickey                                                             */
/* load a new mouse cursor into the cursor buffer                         */
/* ---------------------------------------------------------------------- */
void MakeMickey(void)
{

    /* accumulate color value and position to cursor array     */
    /* each pixel value in the CGA MED RES is stored in 2 bits */
    /* and the mouse cursor mask is stored in a word value     */
    /* the pixel map is defined by a left shift of the pixel   */
    /* attribute to the appropriate position in the mask.      */

    int shifter,i,j,k=16;

    for(i=0;i<16;i++){
        for(j=0;j<8;j++){
            switch(j){
                case 0: shifter=14;break;
                case 1: shifter=12;break;
                case 2: shifter=10;break;
                case 3: shifter= 8;break;
                case 4: shifter= 6;break;
                case 5: shifter= 4;break;
                case 6: shifter= 2;break;
                case 7: shifter= 0;break;
             }
            Cursor[k]=(Cursor[k] | Arrow[i][j]<<shifter);
            Cursor[i]=(Cursor[i] | Zorro[i][j]<<shifter);
        }
        k++;
    }

}


/* ---------------------------------------------------------------------- */
/* HiMickey                                                               */
/* Check for a valid mouse driver and mouse.                              */
/* ---------------------------------------------------------------------- */
int HiMickey()
{
    union REGS inregs, outregs;
    struct SREGS segregs;
    long address;
    unsigned char first_byte;

    /* check for microsoft compatible mouse driver */
    inregs.x.ax = 0x3533;
    intdosx ( &inregs, &outregs, &segregs );
    address = (((long) segregs.es) << 16) + (long) outregs.x.bx;
    first_byte = * (long far *) address;
    if ((address == 0) || (first_byte == 0xcf)) return FALSE;
    else bMickeyMouse = TRUE;

    /* check to see if the mouse is responding */
    inregs.x.ax=0;
    int86(0x33,&inregs,&outregs);
    if(outregs.x.ax==0)return FALSE;

    /* load the cursor */
    segread(&segregs);
    segregs.es=segregs.ds;
    inregs.x.ax=9;
    inregs.x.bx=4;
    inregs.x.cx=8;
    inregs.x.dx=*((unsigned int*)&CursorPtr);
    int86x(0x33,&inregs,&outregs,&segregs);

    return bMickeyMouse;
}
