#include <errno.h>
#include <fnmatch.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "ftn.h"
#include "acl.h"
#include "lutil.h"
#include "xutil.h"

int acl_default_action = PERMIT;

void tidy_acl(aclp)
	acl *aclp;
{
	acl *taclp;

	while (aclp != NULL) {
		if (aclp->etag != NULL) free(aclp->etag);
		if (aclp->from != NULL) free(aclp->from);
		if (aclp->to != NULL) free(aclp->to);
		taclp = aclp->next;
		free(aclp);
		aclp = taclp;
	}
}

acl *read_acl(filename)
	char *filename;
{
	int action;
	int linecount;
	FILE *fp;
	acl *aclp, *taclp;
	char action_s[256], etag_s[256], from_s[256], to_s[256];
	char buf[1024];

	if (filename == NULL) return NULL;

	if ((fp = fopen(filename, "r")) == NULL) {
		if (errno != ENOENT)
			logerr("$cannot open access list \"%s\"", filename);
		return NULL;
	}

	linecount = 1;
	aclp = taclp = NULL;

	while (fgets(buf, sizeof(buf)-1, fp) != NULL) {
		if (buf[0] == '#') continue;

		if (sscanf(buf, "%s%s%s%s",
		    action_s, etag_s, from_s, to_s) != 4) {
			logerr("error in access list \"%s\", line %d",
			    filename, linecount);
			continue;
		}

		if (strcmp(action_s, "deny") == 0)
			action = DENY;
		else if (strcmp(action_s, "permit") == 0)
			action = PERMIT;
		else {
			logerr("bad action in access list \"%s\", line %d",
			    filename, linecount);
			continue;
		}

		if (taclp == NULL)
			aclp = taclp = (acl *)xmalloc(sizeof(acl));
		else {
			taclp->next = (acl *)xmalloc(sizeof(acl));
			taclp = taclp->next;
		}
			
		taclp->action = action;
		taclp->etag = xstrcpy(etag_s);
		taclp->from = xstrcpy(from_s);
		taclp->to = xstrcpy(to_s);
		taclp->next = NULL;
	}

	fclose(fp);
	return aclp;
}

int match_acl(etag, from, to, aclp)
	char *etag;
	faddr *from;
	faddr *to;
	acl *aclp;
{
	int action = acl_default_action;
	char *etag_s = NULL, *from_s = NULL, *to_s = NULL;

	if (etag != NULL) {
		etag_s = xstrcpy(etag);
		if (etag_s[strlen(etag)-1] == '\n')
			etag_s[strlen(etag)-1] = '\0';
	}

	if (from != NULL)
		from_s = xstrcpy(ascfnode(from, 0x0f));
		
	if (to != NULL)
		to_s = xstrcpy(ascfnode(to, 0x0f));
		

	for (; aclp != NULL; aclp = aclp->next)
		if ((etag == NULL ||
		    fnmatch(aclp->etag, etag_s, FNM_NOESCAPE) == 0) &&
		    (from == NULL ||
		    fnmatch(aclp->from, from_s, FNM_NOESCAPE) == 0) &&
		    (to == NULL ||
		    fnmatch(aclp->to, to_s, FNM_NOESCAPE) == 0)) {
			action = aclp->action;
			break;
		}

	if (etag_s != NULL) free(etag_s);
	if (from_s != NULL) free(from_s);
	if (to_s != NULL) free(to_s);

	return action;
}

#ifdef DEBUG

int main(int argc, char **argv)
{
	int i;
	acl *aclp;

	aclp = read_acl(argv[1]);

	/*for (i=0; i<1000000; i++)
		match_acl(argv[2], argv[3], argv[4], aclp);
	*/

	printf("%d\n", match_acl(argv[2], argv[3], argv[4], aclp));

	return 0;
}

#endif
